# -*- coding: utf-8 -*-
from importlib import import_module
from os import environ as env
from logging import getLogger

from gevent import sleep, spawn

from connector import resources
from util.wait_group import WaitGroup
from yabus.providers import carrier_matching_provider, supplier_provider, point_matching_provider, carrier_provider, \
    register_type_provider, point_relations_provider
from yabus.util import monitoring


__all__ = ['get_resources', 'get_services', 'get_setup']
logger = getLogger(__name__)


def get_resources():
    connector_route_prefix = '/<connector_name>'
    return [
        type(cls.__name__, (cls,), {
            'routes': tuple(connector_route_prefix + x for x in cls.routes)
        }) for cls in [
            resources.Endpoints,
            resources.Segments,
            resources.RawSegments,
            resources.Search,
            resources.BookParams,
            resources.CalcPromo,
            resources.Book,
            resources.Ride,
            resources.RideVariation,
            resources.Confirm,
            resources.Order,
            resources.CalcRefund,
            resources.Refund,
            resources.Cancel,
            resources.TicketBlank,
        ]
    ] + [
        resources.Ping
    ]


def get_services():
    return []


def create_client(clients, connector_name, wait_group):
    while True:
        try:
            module = import_module('.'.join(['yabus', connector_name.partition('-')[0]]))
            module.converter.point_converter.setup()

            clients[connector_name] = module.Client()

            monitoring.set_gauge('clients.count', len(clients))

            return
        except Exception:
            logger.error("Failed init {} client".format(connector_name), exc_info=True)
        finally:
            wait_group.done()
        sleep(5)


def setup_app(app):
    monitoring.init(**app.config['MONITORING'])

    supplier_provider.setup()
    carrier_provider.setup()
    carrier_matching_provider.setup()
    point_relations_provider.setup()
    point_matching_provider.setup()
    register_type_provider.setup()

    # setup singleton objects
    app.config['CLIENTS'] = {}
    connector_names = map(str.strip, env['API_RESOURCES'].split(','))
    wait_group = WaitGroup(len(connector_names))
    for client_name in connector_names:
        spawn(create_client, app.config['CLIENTS'], client_name, wait_group)
    wait_group.wait()


def get_setup():
    return setup_app
