# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from functools import wraps

from flask import Response, current_app as app
from future.utils import raise_from

from util.abort import abort
from util.resource import Resource
from yabus.common.client import Client
from yabus.common.exceptions import Error, PartnerError
from yabus.common.fields import ValidationError
from yabus.util import dump_response
from yabus.util.connector_context import ConnectorContext, connector_context
from yabus.util.explainer import ProtocolExplainer, ExplainedException

logger = logging.getLogger(__name__)


def explain_decorator(method):
    @wraps(method)
    def wrapper(*args, **kwargs):
        explain = kwargs.pop('explain')
        if not explain:
            return method(*args, **kwargs)

        connector_context.explainer = ProtocolExplainer()
        try:
            result = method(*args, **kwargs)
        except Exception as exc:
            raise_from(connector_context.explainer.explained_exception(), exc)

        if isinstance(result, Response):
            abort(400, message='Cannot explain {!r}'.format(result))

        return {
            'result': result,
            'explain': connector_context.explainer.explain(),
        }
    return wrapper


class ConnectorResource(Resource):

    method_decorators = [explain_decorator]

    def __init__(self, *args, **kwargs):
        kwargs.setdefault('params', {})['explain'] = {}
        super(ConnectorResource, self).__init__(*args, **kwargs)

    def dispatch_request(self, connector_name, **kwargs):
        self._connector_name = connector_name  # TODO: use connector_context.connector instead
        abort_kwargs = {}
        try:
            try:
                with ConnectorContext(connector_name):
                    return super(ConnectorResource, self).dispatch_request(**kwargs)
            except ExplainedException as expl:
                abort_kwargs['explain'] = expl.explain
                raise expl.__cause__
        except ValidationError as e:
            abort(400, error=e, message=str(e), **abort_kwargs)
        except PartnerError as e:
            logger.error(dump_response(e.response))
            abort(502, error=e, **abort_kwargs)
        except Error as e:
            if e.context is not None:
                logger.error(e.context)
            abort(422, error=e, **abort_kwargs)

    @property
    def client(self):
        # type: () -> Client
        try:
            return app.config['CLIENTS'][self._connector_name]
        except KeyError:
            abort(404)
