# -*- coding: utf-8 -*-
import logging
from collections import namedtuple

import typing
from flask_restful import inputs
from six.moves import map, reduce

from connector.resources.connector_resource import ConnectorResource
from yabus.common import Identifier
from yabus.common.entities.ticket import STATUS_BOOKED
from yabus.common.exceptions import BookingError
from yabus.util import monitoring

logger = logging.getLogger(__name__)


class PassengerMark(namedtuple('PassengerMark', (
    'lastName', 'middleName', 'firstName', 'birthDate', 'docNumber', 'genderType', 'ticketType',
))):
    @staticmethod
    def _safe_get(values, path):
        # type: (typing.Dict[typing.Text, typing.Any], typing.Text) -> typing.Text
        return reduce(lambda d, k: d.get(k) or {}, path.split('.'), values) or ''

    @classmethod
    def from_book_passenger(cls, passenger):
        # type: (typing.Dict) -> PassengerMark
        return cls(
            cls._safe_get(passenger, 'lastName'),
            cls._safe_get(passenger, 'middleName'),
            cls._safe_get(passenger, 'firstName'),
            cls._safe_get(passenger, 'birthDate')[:10],
            cls._safe_get(passenger, 'docSeries') + cls._safe_get(passenger, 'docNumber'),
            cls._safe_get(passenger, 'genderTypeName'),
            cls._safe_get(passenger, 'ticketTypeName')
        )

    @classmethod
    def from_order_ticket(cls, ticket):
        # type: (typing.Dict) -> PassengerMark
        ticket_passenger = ticket['passenger']
        return cls(
            cls._safe_get(ticket_passenger, 'lastName'),
            cls._safe_get(ticket_passenger, 'middleName'),
            cls._safe_get(ticket_passenger, 'firstName'),
            cls._safe_get(ticket_passenger, 'birthDate')[:10],
            cls._safe_get(ticket_passenger, 'docNumber').replace(' ', ''),
            cls._safe_get(ticket_passenger, 'genderType.name'),
            cls._safe_get(ticket_passenger, 'ticketType.name')
        )

    @classmethod
    def iter_values(cls, book_passenger_mark, order_passenger_mark):
        # type: (typing.Iterable[PassengerMark], typing.Iterable[PassengerMark]) -> typing.Iterable[typing.Tuple[typing.Text, typing.Text, typing.Text]]
        return zip(cls._fields, book_passenger_mark, order_passenger_mark)


def _validate(connector_name, passengers, order):
    order_id = '{}:{}'.format(connector_name, order.get('@id'))
    order_status = order.get('status', {}).get('id')
    if order_status != STATUS_BOOKED:
        raise BookingError(
            'unexpected status for order {}, need {}, got {}'.format(order_id, STATUS_BOOKED, order_status)
        )

    book_passenger_marks = sorted(map(PassengerMark.from_book_passenger, passengers))
    order_passenger_marks = sorted(map(PassengerMark.from_order_ticket, order['tickets']))

    passengers_num = len(book_passenger_marks)
    tickets_num = len(order_passenger_marks)
    if passengers_num != tickets_num:
        raise BookingError(
            'unexpected tickets number for order {}, need {}, got {}'.format(order_id, passengers_num, tickets_num)
        )

    for book_passenger_mark, order_passenger_mark in zip(book_passenger_marks, order_passenger_marks):
        for field_name, book_value, order_value in PassengerMark.iter_values(book_passenger_mark, order_passenger_mark):
            if book_value != order_value:
                monitoring.inc_counter('book.passenger_changes', monitoring.get_context_labels())
                logger.warning(
                    'unexpected passenger data change for order %s, %s: %r -> %r',
                    order_id, field_name, book_value, order_value
                )


class Book(ConnectorResource):
    routes = ('/rides/<ride_id>/book',)

    def __init__(self):
        super(Book, self).__init__(params={
            'passengers': {'type': dict, 'location': 'json', 'action': 'append'},
            'payOffline': {'dest': 'pay_offline', 'type': inputs.boolean, 'default': False},
        })

    def post(self, ride_id, passengers, pay_offline):
        ride_id = Identifier.loads(ride_id)
        order = self.client.book(
            ride_id=ride_id,
            passengers=passengers,
            pay_offline=pay_offline
        )
        try:
            _validate(self._connector_name, passengers, order)
        except BookingError as e:
            logger.error(e.message)
            raise
        except Exception:
            logger.exception('error when validating order of %s', self._connector_name)

        return order
