# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import enum

from collections import namedtuple

from travel.proto.dicts.buses.common_pb2 import EPointKeyType, TPointKey


class PointType(enum.Enum):
    STATION = 's'
    SETTLEMENT = 'c'


class PointKey(namedtuple('PointKey', 'type id')):
    @classmethod
    def load(cls, raw_point_key):
        return cls(PointType(raw_point_key[:1]), int(raw_point_key[1:]))

    @classmethod
    def load_from_proto(cls, point_key_proto):
        if point_key_proto.type == EPointKeyType.POINT_KEY_TYPE_STATION:
            point_type = PointType.STATION
        elif point_key_proto.type == EPointKeyType.POINT_KEY_TYPE_SETTLEMENT:
            point_type = PointType.SETTLEMENT
        else:
            raise ValueError('unknown pointkey type')
        return cls(point_type, point_key_proto.id)

    def dump_to_proto(self):
        if self.type == PointType.STATION:
            point_key_proto_type = EPointKeyType.POINT_KEY_TYPE_STATION
        elif self.type == PointType.SETTLEMENT:
            point_key_proto_type = EPointKeyType.POINT_KEY_TYPE_SETTLEMENT
        else:
            raise ValueError('unknown pointkey type')
        return TPointKey(id=self.id, type=point_key_proto_type)

    def __str__(self):
        return '{}{}'.format(self.type.value, self.id)

    @classmethod
    def station(cls, station_id):
        return cls(PointType.STATION, station_id)

    @classmethod
    def settlement(cls, settlement_id):
        return cls(PointType.SETTLEMENT, settlement_id)
