# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import re
from datetime import datetime

from flask_restful import fields

from yabus import common
from yabus.atlasbus.converter import point_converter
from yabus.atlasbus.identifiers import RideID as IdentifiersRideID
from yabus.common import fields as cfields


class Ride(common.Ride):
    @cfields.id_wrapper(IdentifiersRideID)
    class RideID(cfields.Dict):
        def __init__(self):
            cfields.Dict.__init__(self, ride_sid='id', from_ssid='from.id', to_ssid='to.id',
                                  from_scid='from_scid', to_scid='to_scid', date_str='date_str',
                                  departure='departure', arrival='arrival')

    class DateTime(fields.DateTime):
        def format(self, value):
            value = datetime.strptime(value, '%Y-%m-%dT%H:%M:%S')
            return super(Ride.DateTime, self).format(value)

    @cfields.converter
    class Benefit(common.Ride.BenefitType):
        conv = {
            'air': 7,
            'wifi': 4,
            '220v': 1,
            'coffee': 0,
            'tv': 3,
            'no-ticket-required': 5,
            'child-seat': 9,
        }

    class Status(common.Ride.Status):
        default_status = common.Ride.Status.SALE

    fields = {
        '@id': RideID(),
        'refundConditions': fields.String(attribute='refundConditions'),
        'status': Status(attribute='__not_existent__'),
        'connector': cfields.Constant('atlasbus'),
        'partner': cfields.Constant('atlasbus'),
        'partnerName': cfields.Constant('Atlasbus'),
        'carrier': fields.String(attribute='carrier'),
        'carrierID': fields.String(attribute='carrierID'),
        'freeSeats': fields.Integer(attribute='freeSeats'),
        'departure': DateTime(attribute='departure', dt_format=b'iso8601'),
        'arrival': DateTime(attribute='arrival', dt_format=b'iso8601'),
        'from': common.Ride.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='from.id'),
            desc=fields.String(attribute='from.desc'),
            supplier_id=fields.String(attribute='from.id')
        ),
        'to': common.Ride.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='to.id'),
            desc=fields.String(attribute='to.desc'),
            supplier_id=fields.String(attribute='to.id')
        ),
        'price': cfields.Price(attribute='price'),
        'fee': cfields.Price(attribute='fee'),
        'number': fields.String(attribute='rideNumber'),
        'name': fields.String(attribute='routeName'),
        'bus': fields.String(attribute='bus.model'),
        'benefits': fields.List(Benefit, attribute='benefits'),
        'onlineRefund': fields.Boolean(attribute='onlineRefund'),
        'currency': fields.String(attribute='currency')
    }

    @classmethod
    def format(cls, value):
        instance = super(Ride, cls).format(value)
        benefits = instance.get('benefits')
        if benefits:
            benefits = [b for b in benefits if b]
            instance['benefits'] = benefits
        if not instance.get('carrierModel'):
            carrier_model = cls._get_carrier(value)
            if carrier_model:
                instance['carrierModel'] = carrier_model
        refund_conditions = instance.get('refundConditions')
        instance['refundConditions'] = (
            re.sub(r'<[^>]+>', '', refund_conditions.replace('</li><li>', '. '))
            if refund_conditions else
            common.Ride.refund_conditions
        )
        return instance

    @classmethod
    def _get_carrier(cls, value):
        carrier = value.get('legal')
        if not carrier:
            return None
        return {
            "legalAddress": carrier.get('address'),
            "registerNumber": "",  # TODO: партнер обещал добавить
            "name": carrier.get('name'),
            "actualAddress": carrier.get('address'),
            "legalName": carrier.get('name'),
            "inn": carrier.get('tin'),
            "timetable": common.Ride.default_carrier_timetable,
        }
