# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from functools import partial
from itertools import chain, product

from yabus import common
from yabus.common.exceptions import PartnerError, PointNotFound
from yabus.common.entities import RawSegments
from yabus.util import decode_utf8
from yabus.util.formatting import python
from yabus.util.parallelism import pmap

from yabus.avibus.exceptions import parse_error
from yabus.avibus.soapclient import SoapClient

logger = logging.getLogger(__name__)


class Client(common.Client, SoapClient):
    def __init__(self, converter, entities, **kwargs):
        SoapClient.__init__(self, **kwargs)
        self.converter = converter
        self.entities = entities

    def endpoints(self):
        return [
            self.entities.Endpoint.init(decode_utf8(python(x)))
            for x in self.call('GetBusStops', [])
        ]

    def raw_segments(self):
        return RawSegments.init({
            'fake_segments': True,
            'segments': self._raw_segments(),
        })

    def segments(self):
        return list(self.converter.gen_map_segments(self._raw_segments()))

    def psearch(self, (from_sid, to_sid), date):
        payload = [
            from_sid,
            to_sid,
            date.date().isoformat(),
        ]
        try:
            response = python(self.call('GetTrips', payload)['Elements'])
            return self.entities.Ride.init(response)
        except PartnerError:
            logger.exception('cannot get rides from_sid=%s to_sid=%s date=%s', from_sid, to_sid, date)
            return []

    def search(self, from_uid, to_uid, date, _=False):
        try:
            func = partial(self.psearch, date=date)
            directions = product(*map(self.converter.deprecated_map, [from_uid, to_uid]))
            return list(chain(*pmap(func, directions)))
        except PointNotFound:
            return []

    @parse_error
    def ride_details(self, ride_id):
        sids = self._unwrap_ride_id(ride_id)
        order = self._start_session(*sids)
        return self.entities.RideDetails.init(order)

    @parse_error
    def book(self, ride_id, passengers, pay_offline):
        sids = self._unwrap_ride_id(ride_id)
        order_sid = self._start_session(*sids)['Number']
        tickets = self._fill_tickets(order_sid, passengers)
        self._set_tickets(order_sid, tickets)
        response = self._reserve_order(order_sid)
        return self.entities.Order.init(response, status=1, tickets_fwd={'status': 1})  # status: 1, booked

    @parse_error
    def confirm(self, order_id):
        order_sid = order_id['order_sid']
        cheque_settings = {'ChequeWidth': 80, 'TaggedText': None}
        response = python(self.call('Payment', [order_sid, '', None, {}, cheque_settings]))
        return self.entities.Order.init(response, status=7, tickets_fwd={'status': 7})  # status: 7, sold

    @parse_error
    def refund_info(self, ticket_id):
        ticket_sid, departure_sid = ticket_id['ticket_sid'], ticket_id['departure_sid']
        rorder = self._add_ticket_return(ticket_sid, departure_sid)
        return self.entities.RefundInfo.init(rorder)

    @parse_error
    def refund(self, ticket_id):
        ticket_sid, departure_sid = ticket_id['ticket_sid'], ticket_id['departure_sid']
        rorder = self._add_ticket_return(ticket_sid, departure_sid)
        rorder_sid = rorder['Number']
        rorder = python(self.call('ReturnPayment', [rorder_sid, '', None, {'Elements': None}, {'ChequeWidth': 80}]))
        return self.entities.Refund.init(rorder)

    def _raw_segments(self):
        endpoints = list(chain.from_iterable(self.converter.mapping.values()))
        return list(
            (x, y) for x, y in product(endpoints, endpoints) if x != y
        )

    def _fill_tickets(self, order_sid, passengers):
        tickets = self._fill_tickets_data(order_sid, passengers)
        return self._fill_personal_data(tickets, passengers)

    def _fill_tickets_data(self, order_sid, passengers):
        ticket_seats = map(self.entities.TicketSeat.init, passengers)
        return self._add_tickets(order_sid, ticket_seats)

    def _fill_personal_data(self, tickets, passengers):
        personal_data = {str(p['seatCode']): self.entities.Book.init(p) for p in passengers}
        return [
            dict(ticket, PersonalData=personal_data[ticket['SeatNum']])
            for ticket in tickets
        ]

    def _set_tickets(self, order_sid, tickets):
        return python(self.call('SetTicketData', [order_sid, {'Elements': tickets}])['Tickets'])

    def _start_session(self, ride_sid, from_sid, to_sid, order_sid=None):
        return python(self.call('StartSaleSession', [ride_sid, from_sid, to_sid, order_sid]))

    def _occupied_seats(self, order_sid, trip_sid, from_sid, to_sid):
        occupied_seats = self.call('GetOccupiedSeats', [trip_sid, from_sid, to_sid, order_sid])['return']
        if not occupied_seats:
            return []
        return python(occupied_seats['Elements'])

    def _add_tickets(self, order_sid, ticket_seats):
        return python(self.call('AddTickets', [order_sid, {'Elements': ticket_seats}])['return']['Tickets'])

    def _reserve_order(self, order_sid):
        return python(self.call('ReserveOrder', [order_sid, None, None, {'ChequeWidth': 80}]))

    def _add_ticket_return(self, ticket_sid, departure, rorder_sid=None):
        return python(self.call('AddTicketReturn', [ticket_sid, None, departure, rorder_sid]))

    @staticmethod
    def _unwrap_ride_id(ride_id):
        return ride_id['ride_sid'], ride_id['from_sid'], ride_id['to_sid']
