# -*- coding: utf-8 -*-
from flask_restful import fields

from yabus import common
from yabus.avibus.citizenships import citizenships
from yabus.avibus.entities.ride import Ride
from yabus.common import fields as cfields
from yabus.common.fields import Optional


class RideDetails(common.RideDetails):
    @common.fields.converter
    class DocumentType(common.RideDetails.DocumentType):
        conv = {
            'Паспорт гражданина РФ': 1,
            'Загранпаспорт гражданина РФ': 3,
            'Свидетельство о рождении': 2,
        }

    @common.fields.converter
    class GenderType(common.RideDetails.GenderType):
        conv = {
            'Мужской': 1,
            'Женский': 2,
        }

    @common.fields.converter
    class TicketType(common.RideDetails.TicketType):
        # TODO 'Багажный': 3
        conv = {
            'Пассажирский': 1,
            'Детский': 2,
        }

    _doc_types = [{'code': c, 'type': t} for c, t in DocumentType.variants()]
    _gender_types = [{'code': c, 'type': t} for c, t in GenderType.variants()]
    _citizenships = [{'code': k, 'name': v} for k, v in citizenships.items()]

    fields = {
        'docTypes': fields.Raw(attribute='__not_existent__', default=_doc_types),
        'ticketTypes': fields.List(attribute='Trip.Fares', cls_or_instance=fields.Nested({
            'code': fields.String(attribute='Caption'),
            'type': TicketType(attribute='Caption'),
            'price': fields.Float(attribute='Cost'),
        })),
        'genderTypes': fields.Raw(attribute='__not_existent__', default=_gender_types),
        'citizenships': fields.Raw(attribute='__not_existent__', default=_citizenships),
        'seats': fields.List(attribute='Trip.Bus.SeatsScheme', cls_or_instance=fields.Nested({
            'code': fields.String(attribute='SeatNum'),
            'number': fields.String(attribute='SeatNum'),
        })),
        'departure': fields.DateTime(attribute='DepartureTime', dt_format='iso8601'),
        'arrival': Ride.ArrivalTime(attribute='ArrivalTime', dt_format='iso8601'),
    }

    @classmethod
    def format(cls, value):
        instance = super(RideDetails, cls).format(value)

        occupied_seats = {s['Number'] for s in (value.get('OccupiedSeats') or []) if s['Type'] == 'Passenger'} | {None, "0"}
        instance['seats'] = [s for s in instance['seats'] if s['number'] not in occupied_seats]

        instance['citizenships'] = [x for x in instance['citizenships'] if x['name']]

        return instance
