# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import typing
from collections import namedtuple
from datetime import datetime

from gevent.lock import RLock
from travel.library.python.rasp_vault.api import get_secret

import config
from yabus import common
from yabus.busfor import defaults
from yabus.busfor.session import Session


class AccessToken(namedtuple("AccessToken", "value expires_at")):
    def is_expired(self):
        # type: () -> bool
        return datetime.utcnow() >= self.expires_at


class TokenProvider(object):
    _access_token = None  # type: typing.Optional[AccessToken]

    def __init__(self, session_factory, login=None, password=None):
        # type: (typing.Callable[[], Session], typing.Optional[typing.Text], typing.Optional[typing.Text]) -> None
        self._lock = RLock()
        self._session_factory = session_factory
        self._login = login
        self._password = password

    def _get_credentials(self):
        # type: () -> (typing.Text, typing.Text)
        login, password = self._login, self._password
        if login is None or password is None:
            secrets = get_secret(config.CONNECTOR_SECRET_NAME)
            login = self._login = login or secrets["busfor-login"]
            password = self._password = password or secrets["busfor-token"]
        return login, password

    def get_token(self):
        # type: () -> typing.Text
        with self._lock:
            login, password = self._get_credentials()
            access_token = self._access_token
            if access_token is None or access_token.is_expired():
                with self._session_factory() as session:
                    token_data = session.post("v2/login", data={"username": login, "password": password})
                access_token = self._access_token = AccessToken(
                    token_data["access_token"],
                    datetime.utcfromtimestamp(token_data["expires_on"]) - defaults.ACCESS_TOKEN_EXPIRATION_OFFSET,
                )

        return access_token.value


class BaseClient(common.HttpClient):
    session_cls = Session
    token_provider_cls = TokenProvider

    def __init__(self, login=None, password=None, *args, **kwargs):
        super(BaseClient, self).__init__()

        token_provider = self.token_provider_cls(lambda: self.session_cls(*args, **kwargs), login, password)
        self.session = lambda: self.session_cls(token=token_provider.get_token(), *args, **kwargs)
