# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from flask_restful import fields

from yabus import common
from yabus.busfor import defaults

logger = logging.getLogger(__name__)


class Endpoint(common.Endpoint):
    COUNTRY_TYPE_ID = 1
    REGION_SUBTYPE_ID = 6
    DISTRICT_SUBTYPE_ID = 7
    COUNTRY_CODE_TYPE_ID = 3  # GENC_2A_Code (https://evs.nci.nih.gov/ftp1/GENC/Archive/ISO-GENC%20Comparison.xlsx)

    LOCATION_PREFIX = "L"
    POINT_PREFIX = "P"

    fields = {
        "supplier_id": fields.String(),
        "type": common.Endpoint.Type(),
        "title": fields.String(),
        "description": fields.String(),
        "latitude": fields.Float(),
        "longitude": fields.Float(),
        "country": fields.String(),
        "country_code": fields.String(),
        "city_id": fields.String(attribute="location_id"),
        "region": fields.String(),
        "district": fields.String(),
        "timezone_info": fields.String(attribute="timezone"),
        "extra_info": fields.String(),
    }

    @classmethod
    def enrich(cls, obj, is_location=False, parents=None, subtypes=None, **kwargs):
        obj = super(Endpoint, cls).enrich(obj, **kwargs)
        obj.update(cls._get_location_fields(obj, parents, subtypes) if is_location else cls._get_point_fields(obj))
        return obj

    @staticmethod
    def _get_translation(obj):
        return next((t for t in (obj.get("translations") or ()) if t["lang"] == defaults.LANGUAGE), None)

    @staticmethod
    def _iter_parents(obj, parents):
        if not parents:
            return

        while True:
            parent_id = obj.get("parent_id")
            if parent_id is None:
                return
            try:
                parent = parents[parent_id]
            except KeyError:
                logger.warning("Parent %s not found for location %s", parent_id, obj["id"])
                return
            yield parent
            obj = parent

    @classmethod
    def _format_location_title(cls, location, location_name, subtypes):
        subtype_id = location["subtype_id"]
        subtype_name = subtypes.get(subtype_id, {}).get("short_name")
        if subtype_name:
            return (
                "{} {}".format(location_name, subtype_name)
                if subtype_id in {cls.REGION_SUBTYPE_ID, cls.DISTRICT_SUBTYPE_ID}
                else "{} {}".format(subtype_name, location_name)
            )

        return location_name

    @classmethod
    def _get_location_fields(cls, obj, parents, subtypes):
        translation = cls._get_translation(obj) or {}

        result = {
            "supplier_id": "{}{}".format(cls.LOCATION_PREFIX, obj["id"]),
            "type": "city",
            "title": cls._format_location_title(obj, translation.get("name") or obj["default_name"], subtypes),
            "country": translation.get("country_name"),
            "region": translation.get("region_name"),
        }

        description_parts = []
        for parent in cls._iter_parents(obj, parents):
            parent_translation = cls._get_translation(parent) or {}
            parent_title = cls._format_location_title(
                parent, parent_translation.get("name") or parent["default_name"], subtypes
            )

            description_parts.append(parent_title)

            if parent["type_id"] == cls.COUNTRY_TYPE_ID:
                result.update(
                    {
                        "country": parent_title,
                        "country_code": next(
                            (d["value"] for d in parent["location_data"] if d["id"] == cls.COUNTRY_CODE_TYPE_ID), None,
                        ),
                    }
                )
            elif parent["subtype_id"] == cls.REGION_SUBTYPE_ID:
                result["region"] = parent_title
            elif parent["subtype_id"] == cls.DISTRICT_SUBTYPE_ID:
                result["district"] = parent_title

        result["description"] = ", ".join(description_parts)

        return result

    @classmethod
    def _get_point_fields(cls, obj):
        translation = cls._get_translation(obj) or {}
        return {
            "supplier_id": "{}{}".format(cls.POINT_PREFIX, obj["id"]),
            "type": "station",
            "title": translation.get("name") or obj["default_name"],
            "description": translation.get("address") or obj.get("default_address"),
        }
