# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import datetime

from flask_restful import fields
from tzlocal import get_localzone as localtz

from yabus import common
from yabus.busfor.defaults import SUPPLIER_CODE
from yabus.busfor.entities.ride import Ride
from yabus.busfor.entities.ticket import Ticket
from yabus.busfor.identifiers import OrderID
from yabus.common import fields as cfields
from yabus.common.entities.ticket import STATUS_RETURNED, STATUS_SOLD, STATUS_UNKNOWN


class Order(common.Order):

    @cfields.id_wrapper(OrderID)
    class OrderID(cfields.Dict):
        pass

    class TotalPrice(Ride.Price):
        def format(self, value):
            total_price = 0
            for ticket in value:
                total_price += ticket['total_amount']
            return super(Order.TotalPrice, self).format(total_price)

    class ExpirationDateTime(fields.DateTime):
        def format(self, value):
            buyout_delta = value[0].get('minutes_for_buyout', 15)
            buyout_datetime = datetime.datetime.now(localtz()) + datetime.timedelta(minutes=buyout_delta)
            return super(Order.ExpirationDateTime, self).format(buyout_datetime)

    # TODO fee? total_agent_income in ticket?
    fields = {
        '@id': OrderID(order_sid='order_id'),
        'partner': cfields.Constant(SUPPLIER_CODE),
        'partnerName': cfields.Constant('Busfor'),
        'supplierId': fields.String(attribute='order_id'),
        'status': common.Ticket.Status,
        'price': TotalPrice(attribute='tickets'),
        'tickets': fields.List(Ticket, attribute='tickets'),
        'expirationDateTime': ExpirationDateTime(attribute='tickets', dt_format=b'iso8601'),
    }

    @classmethod
    def format(cls, value):
        instance = super(Order, cls).format(value)
        if not instance['status'] or instance['status']['name'] == 'unknown':
            instance['status'] = cls._get_order_status(instance['tickets'])
        return instance

    @classmethod
    def _get_order_status(cls, tickets):
        statuses = set()
        for ticket in tickets:
            statuses.add(ticket['status']['id'])

        status = common.Ticket.Status()
        if len(statuses) == 1:
            return status.format(statuses.pop())
        if statuses == {STATUS_SOLD, STATUS_RETURNED}:
            return status.format(STATUS_SOLD)

        return status.format(STATUS_UNKNOWN)
