# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import datetime

from flask_restful import fields

from yabus import common
from yabus.busfor.converter import point_converter
from yabus.busfor.entities.endpoint import Endpoint
from yabus.busfor.identifiers import RideID
from yabus.common import fields as cfields


class Ride(common.Ride):
    BASE_TARIFFS = ['базовый', 'полный', 'пассажирский']

    @cfields.id_wrapper(RideID)
    class RideID(cfields.Dict):
        pass

    class DateTime(fields.DateTime):
        def format(self, value):
            value = datetime.strptime(value, '%Y-%m-%d %H:%M')
            return super(Ride.DateTime, self).format(value)

    class Description(fields.String):
        def output(self, key, obj):
            point_id = super(Ride.Description, self).output(key, obj)
            point = obj["points"].get(point_id, {})
            desc = ", ".join(filter(None, [point.get("name"), point.get("address")]))
            if not desc:
                return self.default
            return self.format(desc)

    class Carrier(fields.String):
        def output(self, key, obj):
            return self.format(obj['carriers'][obj[self.attribute]].get('name'))

    @cfields.converter
    class Point(common.Ride.Point):
        def conv(self, value):
            return '{}{}'.format(Endpoint.POINT_PREFIX, value)

    class SupplierPoint(fields.String):
        def format(self, value):
            return super(Ride.SupplierPoint, self).format('{}{}'.format(Endpoint.POINT_PREFIX, value))

    class Price(cfields.Price):
        def format(self, value):
            value = super(Ride.Price, self).format(value)
            return value * 0.01

    class Status(common.Ride.Status):
        def format(self, value):
            return super(Ride.Status, self).format(1 if value else 0)

    class Bus(fields.String):
        def output(self, key, obj):
            return self.format(obj['vehicles'][obj[self.attribute]].get('model'))

    @cfields.converter
    class Benefit(common.Ride.BenefitType):
        conv = {
            '1': 6,  # wc
            '2': 7,  # conder
            '8': 4,  # wi-fi
            '9': 1,  # 220v
            '12': 0,  # drinks
            '13': 8,  # common audio
            '15': 3,  # TV

            # Partner codes:
            # vehicle parameter.Possible values 1. Working WC 2. Climate control 3. Superior seats 4. The seats are
            # pushed back 5. The seats sliding across the width 6. Seat belts on each seat 7. Child seats 8. Wi - fi
            # 9. Sockets 220v 10. USB Sockets 11. Coach crew 12. Drinks and water on the road.
            # 13. Common audio system to the whole bus 14. Individual audio system for each passenger
            # 15. Common video system to the whole bus 16. Video Monitor for each passenger
            # 17. Individual lighting for each passenger
            # 18. VIP lounge (increased distance between rows and / or leather seats, etc.)
            # 19. Free meals 20. Meals (extra charge) 21. GPS location 22. Above standard luggage
            # 23. Space between seats with 73cm or more 24.Female dedicated space 25. Disable accessbility
            # 26. Video; Media; DVD or TV with streaming
        }

    class RefundConditions(fields.String):
        TO_TRUNC = ' ориентировочная сумма'  # remove approximately refund amount from refund conditions

        def format(self, value):
            if len(value) == 1:
                return self._get_refund_conditions(value[0])
            for tariff in value:
                # TODO бывает еще полный и тд, перейти на регулярку
                if any(tariff['name'].strip().lower().startswith(t) for t in Ride.BASE_TARIFFS):
                    return self._get_refund_conditions(tariff)
            return self.default

        def _get_refund_conditions(self, tariff):
            conditions = []
            refund_conditions = tariff.get('refund_conditions')
            if not refund_conditions:
                return self.default
            for condition in refund_conditions:
                cond_desc = condition['condition_description']
                trunc_pos = cond_desc.lower().find(self.TO_TRUNC)
                if trunc_pos > 0:
                    conditions.append(cond_desc[:trunc_pos])
                else:
                    conditions.append(cond_desc)
            return '\n'.join(conditions)

    class Fee(fields.Float):
        def format(self, value):
            partner_fee = 0.0
            for tariff in value:
                commissions = tariff.get('commissions', []) or []
                if any(tariff['name'].strip().lower().startswith(t) for t in Ride.BASE_TARIFFS):
                    for commission in commissions:
                        partner_fee += commission.get('commission_value', 0)
                    return partner_fee * 0.01
            return self.default

    # TODO handle with 'trip_options'
    fields = {
        '@id': RideID(ride_sid='id', ride_date='search_date_str'),
        'status': Status(attribute='sale_enable'),
        'connector': cfields.Constant('busfor'),
        'partner': cfields.Constant('busfor'),
        'partnerName': cfields.Constant('Busfor'),
        'carrier': Carrier(attribute='carrier_id'),
        'carrierID': fields.String(attribute='carrier_id'),
        'freeSeats': fields.Integer(attribute='free_seats'),
        'departure': DateTime(attribute='departureDateTime', dt_format=b'iso8601'),
        'arrival': DateTime(attribute='arrivalDateTime', dt_format=b'iso8601'),
        'from': common.Ride.Endpoint(
            id=Point(converter=point_converter, attribute='departure_point_id'),
            desc=Description(attribute='departure_point_id'),
            supplier_id=SupplierPoint(attribute='departure_point_id')
        ),
        'to': common.Ride.Endpoint(
            id=Point(converter=point_converter, attribute='arrival_point_id'),
            desc=Description(attribute='arrival_point_id'),
            supplier_id=SupplierPoint(attribute='arrival_point_id')
        ),
        'price': Price(attribute='price.total'),
        'fee': Fee(attribute='price.tariffs'),
        'number': fields.String(attribute='number'),
        'name': fields.String(attribute='name'),
        'bus': Bus(attribute='vehicle_id'),
        'benefits': fields.List(Benefit(attribute='option_id'), attribute='bus_options'),
        'onlineRefund': cfields.Constant(True),
        'refundConditions': RefundConditions(attribute='price.tariffs', default=common.Ride.refund_conditions),
        'currency': fields.String(attribute="price.currency")
    }

    @classmethod
    def format(cls, value):
        instance = super(Ride, cls).format(value)
        benefits = instance.get('benefits')
        if benefits:
            benefits = [b for b in benefits if b]
            instance['benefits'] = benefits
        cls._add_own_carrier(instance, value)
        return instance

    @classmethod
    def _add_own_carrier(cls, instance, value):
        if instance.get('carrierModel'):
            return

        carrier_id = value.get('carrier_id')
        carriers = value.get('carriers')
        if not carrier_id or not carriers:
            return
        carrier = carriers.get(carrier_id)
        if not carrier:
            return

        instance['carrierModel'] = {
            "legalAddress": carrier.get('legal_address'),
            "registerNumber": carrier.get('ogrn'),
            "name": carrier.get('name'),
            "actualAddress": carrier.get('physical_address'),
            "legalName": carrier.get('name'),
            "inn": carrier.get('inn'),
            "timetable": common.Ride.default_carrier_timetable,
        }
