# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import re

from flask_restful import fields

from yabus import common
from yabus.busfor.citizenships import citizenships
from yabus.busfor.converter import point_converter
from yabus.busfor.entities.ride import Ride
from yabus.common import fields as cfields


class RideDetails(common.RideDetails):

    @cfields.converter
    class GenderType(common.RideDetails.GenderType):
        conv = {
            'M': 1,
            'F': 2,
        }

    # TODO PROM стоит убрать из фильтра, в поездках бывает только один тариф YPROM, в этом случае падаем
    class TicketType(common.RideDetails.TicketType):
        PATTERN = re.compile(r'Y(?:\d*)(?P<discount>(?:B|Z|G|LL|PCC|SI|S|INF|PROM)*)')

        def format(self, value):
            match = self.PATTERN.match(value)
            if match is None:
                return None

            discount = match.group(b'discount')
            if not discount:
                return super(RideDetails.TicketType, self).format(common.RideDetails.TICKET_TYPE_FULL)
            elif discount == 'B':
                return super(RideDetails.TicketType, self).format(common.RideDetails.TICKET_TYPE_CHILD)

            return None

    class Fee(fields.Float):
        def format(self, commissions):
            return sum(commission.get('commission_value', 0) for commission in commissions) * 0.01

    @cfields.converter
    class Country(fields.Raw):
        conv = citizenships

    @cfields.converter
    class PlaceType(common.RideDetails.PlaceType):
        conv = {
            0: 2,
            1: 1,
            2: 0,
        }

    @cfields.converter
    class SeatStatus(common.RideDetails.SeatStatus):
        conv = {
            1: 0,
            2: 1,
            3: 1,
        }

    class DocumentType(common.RideDetails.DocumentType):
        # 1 - Паспорт
        # 7 - Свидетельство о рождении
        # 4 - Загранпаспорт
        # 1 - Иностранный паспорт, идентификатор совпадает с паспортом, код страны иной
        DOC_TYPE_CODES = {
            '1': [common.RideDetails.DOC_TYPE_PASSPORT, common.RideDetails.DOC_TYPE_FOREIGN_PASSPORT],
            '7': [common.RideDetails.DOC_TYPE_BIRTH_CERT],
            '4': [common.RideDetails.DOC_TYPE_INTERNATIONAL_PASSPORT]}

        def __init__(self, document_type_attr, citizenship_attr, *args, **kwargs):
            super(RideDetails.DocumentType, self).__init__(*args, **kwargs)
            self._citizenship = fields.String(attribute=citizenship_attr)
            self._document_type = fields.String(attribute=document_type_attr)

        def output(self, key, obj):
            doc_type = self._document_type.output(key, obj)
            if doc_type not in self.DOC_TYPE_CODES:
                return self.default
            # if not passport - we have one value to return
            if doc_type != '1':
                return self.format(self.DOC_TYPE_CODES[doc_type][0])
            # if passport - need to check citizenship
            else:
                citizenship = self._citizenship.output(key, obj)
                # TODO оставить один вариант после того, как будет понятно, что шлет басфор
                if citizenship == 'Россия' or citizenship == 'RU':
                    return self.format(common.RideDetails.DOC_TYPE_PASSPORT)
                else:
                    return self.format(common.RideDetails.DOC_TYPE_FOREIGN_PASSPORT)

    FREE_SEAT_STATUS = 1

    class FreeSeats(fields.List):
        def format(self, value):
            return super(RideDetails.FreeSeats, self).format(
                [seat for seat in value if seat['status'] == RideDetails.FREE_SEAT_STATUS])

    _citizenships = [{'code': code, 'name': code} for name, code in Country.variants()]

    fields = {
        'docTypes': cfields.Constant([
            {
                'code': supplier_doc_id,
                'type': common.RideDetails.DocumentType().format(buses_doc_id)
            } for supplier_doc_id, buses_docs_ids in DocumentType.DOC_TYPE_CODES.items()
            for buses_doc_id in buses_docs_ids]),
        'ticketTypes': fields.List(
            fields.Nested(
                {
                    'code': fields.String(attribute='code'),
                    'type': TicketType(attribute='code'),
                    'price': Ride.Price(attribute='cost'),
                    'fee': Fee(attribute='commissions'),
                }
            ),
            attribute='segment.price.tariffs',
        ),
        'genderTypes': cfields.Constant([{'code': c, 'type': t} for c, t in GenderType.variants()]),
        'citizenships': fields.Raw(attribute='__not_existent__', default=_citizenships),
        'seats': FreeSeats(fields.Nested({
                'code': fields.String(attribute='id'),
                'number': fields.String(attribute='num'),
        }), attribute='map_seat.map_seat',
        ),
        'map': fields.List(
            fields.Nested(
                {
                    'x': fields.Integer(attribute='x'),
                    'y': fields.Integer(attribute='y'),
                    'type': PlaceType(attribute='seat_type'),
                    'status': SeatStatus(attribute='status'),
                    'number': fields.String(attribute='num'),
                }
            ),
            attribute='map_seat.map_seat',
        ),
        'departure': Ride.DateTime(attribute='segment.departureDateTime', dt_format=b'iso8601'),
        'arrival': Ride.DateTime(attribute='segment.arrivalDateTime', dt_format=b'iso8601'),
        'from': common.RideDetails.Endpoint(
            id=Ride.Point(converter=point_converter, attribute='segment.departure_point_id'),
            desc=Ride.Description(attribute='segment.departure_point_id'),
        ),
        'to': common.RideDetails.Endpoint(
            id=Ride.Point(converter=point_converter, attribute='segment.arrival_point_id'),
            desc=Ride.Description(attribute='segment.arrival_point_id'),
        ),
        'currency': fields.String(attribute='segment.price.currency'),
        'price': Ride.Price(attribute='segment.price.total'),
    }

    @classmethod
    def format(cls, value):
        instance = super(RideDetails, cls).format(value)

        value_map_seat = value.get('map_seat', {})
        # remove place choice if free seating available or got empty map_seat
        if not value_map_seat or value_map_seat.get('is_free_seating', False):
            instance['map'] = None
            instance['seats'] = None
        elif any(seat['z'] != 0 for seat in value_map_seat.get('map_seat', ())):
            # hide map of multi-tier buses
            instance['map'] = None

        return instance
