# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from flask import current_app as app
from flask_restful import fields

from yabus import common
from yabus.busfor.defaults import MIDDLENAME_CAP
from yabus.busfor.entities.ride import Ride
from yabus.busfor.entities.ride_details import RideDetails
from yabus.busfor.identifiers import TicketID
from yabus.common import fields as cfields


class Ticket(common.Ticket):
    @cfields.id_wrapper(TicketID)
    class TicketID(cfields.Dict):
        pass

    @cfields.converter
    class Status(common.Ticket.Status):
        conv = {
            -1: 4,  # ticket processing error
            0: 3,  # in the process of making a return
            1: 1,  # the ticket is reserved
            2: 7,  # ticket purchased
            4: 6,  # ticket returned
            5: 2,  # (not in docs) sometimes when confirm timeouts
            6: 2,  # ticket canceled
            8: 2,  # error from the carrier API
            # 16. PT tariff (boarding pass sale)
        }

    @cfields.converter
    class GenderType(common.RideDetails.GenderType):
        conv = {
            "Мужской": 1,
            "Женский": 2,
        }

    class Document(fields.String):
        def format(self, value):
            doc_num = [x.strip() for x in value.split('/')]
            if len(doc_num) == 2:
                return ' '.join([doc_num[0], doc_num[1]])
            if len(doc_num) == 1:
                return doc_num[0]
            return None

    class Revenue(fields.Float):
        DISPOSITIONS = {
            1,  # the fee is located inside the tariff
            2,  # the fee is located above the tariff
        }
        TYPES = {
            3,  # agent fee №2
            6,  # service charge
        }

        def format(self, commissions):
            revenue_commission = sum(
                (commission.get('value') or 0)
                for commission in (commissions or [])
                if commission.get('disposition') in self.DISPOSITIONS and commission.get('type') in self.TYPES
            )
            if not revenue_commission:
                return self.default

            return round(revenue_commission * 0.01 * app.config['BUSFOR_FEES_REVENUE_RATE'], 2)

    class MiddleName(fields.String):
        def format(self, value):
            return super(Ticket.MiddleName, self).format(value if value != MIDDLENAME_CAP else '')

    # TODO use VAT and feeVat?
    fields = {
        '@id': TicketID(ticket_sid='ticket_id', order_sid='order_id'),
        'supplierId': fields.String(attribute='ticket_number'),
        'status': Status(attribute='ticket_status'),
        'url': fields.String,
        'price': Ride.Price(attribute='total_amount'),
        'revenue': Revenue(attribute='commissions.commissions'),
        'passenger': {
            'firstName': fields.String(attribute='passenger.firstname'),
            'middleName': MiddleName(attribute='passenger.middlenames'),
            'lastName': fields.String(attribute='passenger.lastname'),
            'birthDate': cfields.ISODate(attribute='passenger.birthday', dt_format=b'iso8601'),
            'genderType': GenderType(attribute='passenger.gender'),
            'docType': RideDetails.DocumentType(document_type_attr='passenger.documenttype',
                                                citizenship_attr='passenger.citizenship'),
            'docNumber': Document(attribute='passenger.document'),
            'ticketType': RideDetails.TicketType(attribute='tariff_code'),
            'citizenship': RideDetails.Country(attribute='passenger.citizenship'),
            'seat': fields.String(attribute='number_seat', default='0'),
        },
        'cpaPayload': {
            'commissions': fields.Raw(attribute='commissions.commissions')
        }
    }
