# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import functools
import logging
import types

from six import with_metaclass
from ylog.context import log_context

from yabus.common.pointconverter import PointConverter
from yabus.common.fields import ValidationErrorBundle, ValidationWarning
from yabus.util.connector_context import connector_context

logger = logging.getLogger(__name__)


class ClientMeta(type):
    @staticmethod
    def handle_bundle(bundle, warn_exceptions):
        logger.warning(str(bundle))
        crit = next((e for e in bundle.exceptions if not isinstance(e, warn_exceptions)), None)
        if crit:
            raise crit

        # FIXME: data from some inner Entity.init with ValidationErrorBundle
        # exception should not spoof outer method return value
        return bundle.instance

    @classmethod
    def validation_intercepter(meta, method):
        @functools.wraps(method)
        def wrapper(self, *args, **kwargs):
            try:
                return method(self, *args, **kwargs)
            except ValidationErrorBundle as e:
                warn_exceptions = getattr(method, 'warn_exceptions', self.warn_exceptions)
                return meta.handle_bundle(e, warn_exceptions)

        return wrapper

    @staticmethod
    def connector_method_context(method, method_name):
        @functools.wraps(method)
        def wrapper(self, *args, **kwargs):
            with log_context(method_name=method_name):
                if connector_context:
                    connector_context.method = method_name
                return method(self, *args, **kwargs)

        return wrapper

    _client_method_names = None

    def __new__(meta, name, bases, attrs):
        client_method_names = meta._client_method_names
        if client_method_names is None:
            # caching Client methods names in the first call of __new__
            client_method_names = meta._client_method_names = frozenset(
                attr_name for attr_name, attr in attrs.items() if isinstance(attr, types.FunctionType)
            )

        for method_name, method in tuple(attrs.items()):
            if method_name in client_method_names:
                method = meta.validation_intercepter(method)
                method = meta.connector_method_context(method, method_name)
                attrs[method_name] = method

        return super(ClientMeta, meta).__new__(meta, name, bases, attrs)


class Client(with_metaclass(ClientMeta)):
    """Describes clients interface"""

    warn_exceptions = ValidationWarning

    @property
    def converter(self):
        # type: () -> PointConverter
        raise NotImplementedError

    def endpoints(self):
        """List of all endpoints.
            :return: list of instances of :class:`yabus.common.entities.Endpoint`
        """
        raise NotImplementedError

    def segments(self):
        """List of all possible segments of all routes.
            :return: list of pairs (from-uid, to-uid)
        """
        raise NotImplementedError

    def raw_segments(self):
        """List of all possible partners segments of all routes.
            :return: list of pairs (from-sid, to-sid)
        """
        raise NotImplementedError

    def search(self, from_uid, to_uid, date, try_no_cache=False):
        """Search for a rides.
            :param from_uid: departure point's universal ID
            :param to_uid: arrival point's universal ID
            :param date: date
            :param try_no_cache: call not cached search if available
            :return: list of :class:`yabus.common.entities.Ride`

        .. seealso::
            * :class:`yabus.common.entities.Ride`
        """
        raise NotImplementedError

    def ride(self, ride_id):
        """Get ride with given `ride_id`.
            :param ride_id: instance of :class:`yabus.common.identifiers.Identifier`
            :return: instance of :class:`yabus.common.entities.Ride`

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
            * :class:`yabus.common.entities.Ride`
        """
        raise NotImplementedError

    def ride_details(self, ride_id):
        """Get details for ride with given `ride_id`.
            :param ride_id: instance of :class:`yabus.common.identifiers.Identifier`
            :return: instance of :class:`yabus.common.entities.RideDetails`

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
            * :class:`yabus.common.entities.RideDetails`
        """
        raise NotImplementedError

    def book(self, ride_id, passengers, pay_offline):
        """Book an order.
            :param ride_id: instance of :class:`yabus.common.identifiers.Identifier`
            :param passengers: list of :class:`yabus.common.entities.Book`
            :param pay_offline: offline payment (without online confirmation)
            :return: instance of :class:`yabus.common.entities.Order`

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
            * :class:`yabus.common.entities.Book`
            * :class:`yabus.common.entities.Order`
        """
        raise NotImplementedError

    def confirm(self, order_id):
        """Confirm order.
            :param order_id: instance of :class:`yabus.common.identifiers.Identifier`
            :return: instance of :class:`yabus.common.entities.Order`

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
            * :class:`yabus.common.entities.Order`
        """
        raise NotImplementedError

    def order(self, order_id):
        """Get order.
            :param order_id: instance of :class:`yabus.common.identifiers.Identifier`
            :return: instance of :class:`yabus.common.entities.Order`

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
            * :class:`yabus.common.entities.Order`
        """
        raise NotImplementedError

    def refund_info(self, ticket_id):
        """Get refund info about ticket.
            :param ticket_id: instance of :class:`yabus.common.identifiers.Identifier`
            :return: instance of :class:`yabus.common.entities.RefundInfo`

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
            * :class:`yabus.common.entities.RefundInfo`
        """
        raise NotImplementedError

    def refund(self, ticket_id):
        """Refund ticket.
            :param ticket_id: instance of :class:`yabus.common.identifiers.Identifier`
            :return: instance of :class:`yabus.common.entities.Refund`

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
            * :class:`yabus.common.entities.Refund`
        """
        raise NotImplementedError

    def ticket(self, ticket_id):
        """Get ticket info.
            :param ticket_id: instance of :class:`yabus.common.identifiers.Identifier`
            :return: instance of :class:`yabus.common.entities.Ticket`

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
            * :class:`yabus.common.entities.Ticket`
        """
        raise NotImplementedError

    def ticket_blank(self, ticket_id):
        """Get ticket blank.
            :param ticket_id: instance of :class:`yabus.common.identifiers.Identifier`
            :return: ticket blank file

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
        """
        raise NotImplementedError

    def cancel(self, ticket_id):
        """Cancel ticket.
            :param ticket_id: instance of :class:`yabus.common.identifiers.Identifier`
            :return: instance of :class:`yabus.common.entities.Ticket`

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
            * :class:`yabus.common.entities.Ticket`
        """
        raise NotImplementedError

    def change_ride_endpoints(self, ride_id, pickup_sid, discharge_sid):
        """Change ride endpoints.
            :param ride_id: instance of :class`yabus.common.identifiers.Identifier`
            :param pickup_sid: supplier pickup stop id
            :param discharge_sid: supplier discharge stop id
            :return: instance of :class:`yabus.common.identifiers.Identifier` with new endpoints

        .. seealso::
            * :class:`yabus.common.identifiers.Identifier`
        """
        raise NotImplementedError
