# -*- coding: utf-8 -*-
from yabus.common import fields as cfields
from yabus import util

from flask_restful import fields


class EntityMeta(type):
    def __new__(meta, name, bases, attrs):
        obj_fields = attrs.get('fields', None)
        base_fields = getattr(bases[0], 'fields', None)
        attrs['fields'] = EntityMeta.override(obj_fields, base_fields)
        return super(EntityMeta, meta).__new__(meta, name, bases, attrs)

    @staticmethod
    def override(obj, base):
        def func(obj, base):
            if isinstance(base, cfields.Managed):
                return EntityMeta.override(obj, base.field)
            elif isinstance(base, fields.Nested):
                return EntityMeta.override(obj, base.nested)
            elif isinstance(obj, fields.Nested):
                obj.nested = EntityMeta.override(obj.nested, base)
            elif isinstance(obj, fields.List) and isinstance(base, fields.List):
                obj.container = EntityMeta.override(obj.container, base.container)
            return obj
        return util.merge(obj, base, func)


def marshal_lite(data, fs):
    # Adaptation of the flask_restful marshal with 2 differences:
    # - result is the builtin dict instead of the python-based OrderedDict (30% better performance)
    # - no envelope argument because it is not used
    def make(cls):
        if isinstance(cls, type):
            return cls()
        return cls

    return (
        [marshal_lite(d, fs) for d in data]
        if isinstance(data, (list, tuple)) else
        {k: marshal_lite(data, v) if isinstance(v, dict) else make(v).output(k, data) for k, v in fs.items()}
    )


class Entity(fields.Nested, cfields.Validatable):
    __metaclass__ = EntityMeta

    def __init__(self, **kwargs):
        super(Entity, self).__init__(self.fields, **kwargs)

    @classmethod
    def output(cls, key, obj):
        value = obj if key is None else fields.get_value(key, obj)
        return cls.format(value)

    @classmethod
    def format(cls, value):
        return marshal_lite(value, cls.fields)

    @classmethod
    def validate(cls, value, path):
        return cfields.validate(cls.validation_fields(), value, path)

    @classmethod
    def enrich(cls, obj, **kwargs):
        return util.merge(obj, kwargs)

    @staticmethod
    def _enrich(cls, obj, **kwargs):
        if isinstance(obj, (list, tuple)):
            return [cls.enrich(x, **kwargs) for x in obj]
        return cls.enrich(obj, **kwargs)

    @classmethod
    def init(cls, obj, **kwargs):
        instance = cls.output(None, Entity._enrich(cls, obj, **kwargs))
        exceptions = cls.validate(instance, [cls.__module__, cls.__name__])
        if exceptions:
            raise cfields.ValidationErrorBundle(
                instance=instance, exceptions=exceptions
            )
        return instance

    @classmethod
    def validation_fields(cls):
        while getattr(cls.__base__, 'fields', None) is not None:
            cls = cls.__base__
        return cls.fields
