# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

import six

from flask_restful import fields

from util.proto_to_dict import proto_to_dict
from yabus.carrier_matcher import carrier_matcher
from yabus.common import fields as cfields
from yabus.common.entities.entity import Entity
from yabus.common.fields import Optional, Required, mk_id_name_field
from yabus.providers import supplier_provider, carrier_provider, register_type_provider

logger = logging.getLogger(__name__)

_Status = mk_id_name_field({
    0: 'unknown',
    1: 'sale',
    2: 'canceled',
    3: 'dispatched',
    4: 'closed',
    5: 'boarding',
    6: 'suspended',
})


class Ride(Entity):
    carrier_matcher = carrier_matcher
    supplier_provider = supplier_provider

    class Status(_Status):
        UNKNOWN = 0
        SALE = 1

        default_status = UNKNOWN

        def __init__(self, attribute=None):
            default = _Status.format(self, self.default_status)
            _Status.__init__(self, default=default, attribute=attribute)

    class Endpoint(cfields.Dict):
        def __init__(self, id, desc, supplier_id):
            cfields.Dict.__init__(self, id=id, desc=desc, supplier_id=supplier_id)

    class Point(fields.Raw):
        def __init__(self, converter=None, **kwargs):
            self.converter = converter
            fields.Raw.__init__(self, **kwargs)

        def format(self, value):
            try:
                return self.converter.backmap(six.text_type(value))
            except Exception:
                return self.default

    class BookFields(fields.Raw):
        name = 'name'
        gender = 'gender'
        birth_date = 'birthDate'
        document = 'document'
        phone = 'phone'
        email = 'email'
        __all__ = {name, gender, birth_date, document, phone, email}

        def __init__(self, **kwargs):
            if 'default' not in kwargs:
                kwargs['default'] = self._format(self.__all__)
            fields.Raw.__init__(self, **kwargs)

        def format(self, value):
            return self._format(value)

        def _format(self, value):
            if any(x not in self.__all__ for x in value):
                raise ValueError("Invalid book fields: %r" % value)
            return list(value)

    BenefitType = mk_id_name_field({
        0: 'coffee',
        1: 'charger',
        2: 'press',
        3: 'tv',
        4: 'wi-fi',
        5: 'no-ticket-required',
        6: 'wc',
        7: 'conditioner',
        8: 'common-audio',
        9: 'child-seat',
    })

    refund_conditions = (
        "Возврат на сайте возможен не позднее чем за 15 минут до отправления "
        "автобуса."
    )
    default_carrier_timetable = "пн-пт: 10:00–18:00"

    fields = {
        'connector': Required(fields.String, types=basestring),
        'supplierModel': Required(fields.Raw),
        'partner': Optional(fields.String, types=basestring),
        'partnerName': Required(fields.Raw, types=basestring),
        'partnerPhone': Optional(fields.Raw),
        'partnerEmail': Optional(fields.String, types=basestring),
        '@id': Required(fields.Raw, types=basestring),
        'status': Required(Status, types=dict),
        'number': Optional(fields.Raw),
        'name': Optional(fields.Raw),
        'carrier': Optional(fields.Raw),
        'carrierID': Optional(fields.Raw),
        'carrierModel': Optional(fields.Raw),
        'bus': Optional(fields.Raw),
        'from': Required(
            Endpoint(
                id=Optional(fields.Raw, types=basestring),
                desc=Optional(fields.Raw, types=basestring),
                supplier_id=Optional(fields.Raw, types=basestring),
            ),
            types=dict,
        ),
        'to': Required(
            Endpoint(
                id=Optional(fields.Raw, types=basestring),
                desc=Optional(fields.Raw, types=basestring),
                supplier_id=Optional(fields.Raw, types=basestring),
            ),
            types=dict,
        ),
        'departure': Required(fields.DateTime(dt_format='iso8601'), types=basestring),
        'arrival': Optional(fields.DateTime(dt_format='iso8601'), types=basestring),
        'currency': Required(fields.String(default='RUB', attribute='__not_existent__'), types=basestring),
        'price': Required(fields.Float, types=float),
        'onlinePrice': Optional(fields.Float, types=float),
        'fee': Optional(fields.Float(default=None), types=float),
        'freeSeats': Required(fields.Integer, types=(int, long)),
        'ticketLimit': Required(fields.Integer(default=5, attribute='__not_existent__'), types=(int, long)),
        'refundConditions': Required(fields.Raw(default=refund_conditions), types=basestring),
        'canPayOffline': Required(fields.Boolean(default=False, attribute='__not_existent__'), types=bool),
        'bookOnly': Required(fields.Boolean(default=False, attribute='__not_existent__'), types=bool),
        'onlineRefund': Required(fields.Boolean(default=True, attribute='__not_existent__'), types=bool),
        'bookFields': Required(BookFields(attribute='__not_existent__'), types=list),
        'benefits': Optional(fields.List(BenefitType), types=list),
    }

    @classmethod
    def format(cls, value):
        instance = super(Ride, cls).format(value)
        cls._add_supplier(instance)
        cls._add_carrier(instance)
        return instance

    @classmethod
    def _add_carrier(cls, instance):
        if isinstance(instance, (set, list, tuple)):
            return type(instance)(cls._add_carrier(x) for x in instance)
        elif isinstance(instance, dict):
            if instance['carrierID'] is None:
                instance['carrierID'] = instance['carrier']
            carrier = None
            if instance['carrierID'] is not None:
                carrier = cls.carrier_matcher.get_carrier(
                    supplier_code=six.ensure_text(instance['connector']),
                    carrier_code=six.ensure_text(instance['carrierID']),
                )
            if carrier is None:
                return

            if carrier.timetable == "":
                carrier.timetable = cls.default_carrier_timetable
            instance['carrierModel'] = proto_to_dict(carrier)

            register_type = register_type_provider.get_by_id(carrier.register_type_id)
            if register_type is not None:
                instance['carrierModel']['registerType'] = proto_to_dict(register_type)
        else:
            raise ValueError

    @classmethod
    def _add_supplier(cls, instance):
        if isinstance(instance, (set, list, tuple)):
            return type(instance)(Ride._add_supplier(x) for x in instance)
        elif isinstance(instance, dict):
            supplier = cls.supplier_provider.get_by_code(instance['connector'])
            if supplier is None:
                logger.error('unknown supplier: %s', instance['connector'])
            instance['supplierModel'] = proto_to_dict(supplier)
            for key, val in instance['supplierModel'].items():
                instance['supplierModel'][key] = six.ensure_text(val) if isinstance(val, str) else val

            register_type = register_type_provider.get_by_id(supplier.register_type_id)
            if register_type is not None:
                instance['supplierModel']['registerType'] = proto_to_dict(register_type)
        else:
            raise ValueError
