# -*- coding: utf-8 -*-
from flask_restful import fields

from yabus.common import fields as cfields
from yabus.common.entities.country import Country
from yabus.common.entities.entity import Entity
from yabus.common.fields import Optional, Required, ValidationError, mk_id_name_field


class RideDetails(Entity):
    TICKET_TYPE_FULL = 1
    TICKET_TYPE_CHILD = 2
    TICKET_TYPE_BAGGAGE = 3

    DOC_TYPE_PASSPORT = 1
    DOC_TYPE_BIRTH_CERT = 2
    DOC_TYPE_INTERNATIONAL_PASSPORT = 3
    DOC_TYPE_FOREIGN_PASSPORT = 4
    DOC_TYPE_BELORUSSIAN_PASSPORT = 100

    GENDER_TYPE_MALE = 1
    GENDER_TYPE_FEMALE = 2

    GenderType = mk_id_name_field({
        GENDER_TYPE_MALE: 'male',
        GENDER_TYPE_FEMALE: 'female',
    })

    TicketType = mk_id_name_field({
        TICKET_TYPE_FULL: 'full',
        TICKET_TYPE_CHILD: 'child',
        TICKET_TYPE_BAGGAGE: 'baggage',
    })

    DocumentType = mk_id_name_field({
        # ru
        DOC_TYPE_PASSPORT: 'id',
        DOC_TYPE_BIRTH_CERT: 'birth certificate',
        DOC_TYPE_INTERNATIONAL_PASSPORT: 'passport',
        DOC_TYPE_FOREIGN_PASSPORT: 'foreign passport',
        # by
        DOC_TYPE_BELORUSSIAN_PASSPORT: 'passport.by',
    })

    @cfields.converter
    class OTBDocType(DocumentType):
        """
        https://www.garant.ru/products/ipo/prime/doc/70129008/
        Приказ Минтранса РФ от 19 июля 2012 г. № 243
        Коды документов, удостоверяющих личность, при передаче в АЦБПДП
        """
        conv = {
            0: 1,
            2: 3,
            3: 4,
            4: 2,
        }

    PlaceType = mk_id_name_field({
        0: 'driver',
        1: 'seat',
        2: 'passage',
    })

    SeatStatus = mk_id_name_field({
        0: 'free',
        1: 'occupied',
    })

    stop = fields.Nested({
        'id': Optional(fields.Raw, types=basestring),
        'code': Required(fields.Raw),
        'desc': Optional(fields.Raw, types=basestring),
        'longitude': Optional(fields.Float, types=float),
        'latitude': Optional(fields.Float, types=float),
        'datetime': Optional(fields.DateTime(dt_format='iso8601'), types=basestring),
        'priceDiffs': Required(fields.List(fields.Nested({
            'type': Required(TicketType, types=dict),
            'diff': Required(fields.Float(default=None), types=float),
            'onlineDiff': Optional(fields.Float(default=None), types=float),
        })), types=list),
    })

    class Endpoint(cfields.Dict):
        def __init__(self, id, desc):
            cfields.Dict.__init__(self, id=id, desc=desc)

    fields = {
        'docTypes': fields.List(fields.Nested({
            'code': fields.String,
            'type': Required(DocumentType, types=dict),
        })),
        'ticketTypes': fields.List(fields.Nested({
            'code': fields.String,
            'type': Required(TicketType, types=dict),
            'price': Required(cfields.Price, types=float),
            'fee': Optional(fields.Float, types=float),
            'onlinePrice': Optional(cfields.Price, types=float),
        })),
        'genderTypes': fields.List(fields.Nested({
            'code': fields.String,
            'type': Required(GenderType, types=dict),
        })),
        'citizenships': fields.List(fields.Nested(Country.fields)),
        'seats': Optional(fields.List(fields.Nested({
            'code': fields.String,
            'number': Required(fields.String),
        }))),
        'map': Optional(fields.List(fields.Nested({
            'x': Required(fields.Integer, types=(int, long)),
            'y': Required(fields.Integer, types=(int, long)),
            'type': Required(PlaceType, types=dict),
            'status': Required(SeatStatus, types=dict),
            'number': Required(fields.String, types=basestring)
        }))),
        'pickupStops': Optional(fields.List(stop)),
        'dischargeStops': Optional(fields.List(stop)),
        'from': Optional(
            Endpoint(
                id=Optional(fields.Raw, types=basestring),
                desc=Optional(fields.Raw, types=basestring)
            ),
            types=dict,
        ),
        'to': Optional(
            Endpoint(
                id=Optional(fields.Raw, types=basestring),
                desc=Optional(fields.Raw, types=basestring),
            ),
            types=dict,
        ),
        'departure': Optional(fields.DateTime(dt_format='iso8601'), types=basestring),
        'arrival': Optional(fields.DateTime(dt_format='iso8601'), types=basestring),
        'currency': Optional(fields.String(default='RUB', attribute='__not_existent__'), types=basestring),
        'price': Optional(fields.Float, types=float),
        'fee': Optional(fields.Float, types=float),
    }

    @classmethod
    def format(cls, value):
        instance = super(RideDetails, cls).format(value)
        for types_field in ['docTypes', 'ticketTypes']:
            types = instance.get(types_field, [])
            instance[types_field] = [x for x in types if x.get('type')]

        if not instance['price']:
            for ticket_type in instance.get('ticketTypes', []):
                if ticket_type['type']['name'] == 'full':
                    instance['price'] = ticket_type['price']
                    break
        return instance

    @classmethod
    def validate(cls, instance, path):
        exceptions = super(RideDetails, cls).validate(instance, path)
        if instance['seats'] is not None and not instance['seats']:
            exceptions.append(ValidationError(path + ['seats'], "must not be empty"))
        if not instance['ticketTypes']:
            exceptions.append(ValidationError(path + ['ticketTypes'], "must not be empty"))
        return exceptions
