# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from flask import current_app as app
from flask_restful import fields

from yabus.common import fields as cfields
from yabus.common.entities.entity import Entity
from yabus.common.entities.ride_details import RideDetails
from yabus.common.fields import Optional, Required, ValidationWarning, mk_id_name_field
from yabus.util import pairwise

STATUS_UNKNOWN = 0
STATUS_BOOKED = 1
STATUS_CANCELLED = 2
STATUS_CREATED = 3
STATUS_ERROR = 4
STATUS_PARTIALLY_RETURNED = 5
STATUS_RETURNED = 6
STATUS_SOLD = 7

STATUSES = {
    STATUS_UNKNOWN: 'unknown',
    STATUS_BOOKED: 'booked',
    STATUS_CANCELLED: 'cancelled',
    STATUS_CREATED: 'created',
    STATUS_ERROR: 'error',
    STATUS_PARTIALLY_RETURNED: 'partially returned',
    STATUS_RETURNED: 'returned',
    STATUS_SOLD: 'sold',
}

STATUS_TO_ID = {v: k for k, v in STATUSES.items()}

_Status = mk_id_name_field(STATUSES)


class Ticket(Entity):
    class Status(_Status):
        default_status = 0  # unknown

        def __init__(self, attribute=None):
            default = _Status.format(self, self.default_status)
            _Status.__init__(self, default=default, attribute=attribute)

    fields = {
        '@id': Required(fields.String),
        'supplierId': Required(fields.String),
        'status': Required(Status, types=dict),
        'url': Optional(fields.String),
        'data': Optional(fields.Nested({
            'id': Required(fields.String),
            'code': Optional(fields.String),
            'number': Required(fields.String),
            'series': Required(fields.String),
            'barcode': Optional(fields.String),
            'platform': Optional(fields.String),
        }, allow_null=True), types=dict),
        'price': Required(cfields.Price, types=float),
        'revenue': Optional(fields.Float, types=float),
        'cpaPayload': Optional(fields.Raw),
        'priceVat': Optional(cfields.VAT, types=basestring),
        'feeVat': Optional(cfields.VAT, types=basestring),
        'passenger': fields.Nested({
            'firstName': Required(fields.String),
            'middleName': Optional(fields.String),
            'lastName': Required(fields.String),
            'birthDate': Optional(fields.DateTime(dt_format='iso8601'), types=basestring),
            'genderType': Optional(RideDetails.GenderType, types=dict),
            'citizenship': Optional(fields.String),
            'docType': Optional(RideDetails.DocumentType, types=dict),
            'docNumber': Optional(fields.String),
            'ticketType': Required(RideDetails.TicketType, types=dict),
            'seat': Optional(fields.String(default='0')),
        }),
    }

    @classmethod
    def validate(cls, instance, path):
        exceptions = super(Ticket, cls).validate(instance, path)
        status = instance['status']['id']
        if status == STATUS_SOLD and all(instance.get(k) is None for k in ['data', 'url']):
            exceptions.append(ValidationWarning(path + ['data/url'], "must not be None"))
        doc_fields = (instance['passenger'][x] for x in ['citizenship', 'docType', 'docNumber'])
        if any((x is None) != (y is None) for x, y in pairwise(doc_fields)):
            exceptions.append(
                ValidationWarning(path + ['citizenship/docType/docNumber'], "must be all None or non None"))
        return exceptions

    @classmethod
    def format(cls, value):
        instance = super(Ticket, cls).format(value)
        gen_url = value.pop('__gen_url__', False)
        if gen_url:
            instance['url'] = '{}/{}/tickets/{}/blank'.format(
                 app.config['YANDEX_BUS_API'], value['__partner__'], instance['@id'],
            )
        else:
            instance['url'] = None
        passenger = instance['passenger']
        doc_type = passenger['docType'] or {}
        if not passenger['citizenship'] and doc_type.get('name') in ['id', 'birth certificate', 'passport']:
            passenger['citizenship'] = 'RU'
        return instance
