# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

__all__ = [
    # System errors: code < 100
    'Error',
    'InvalidIdentifier',
    'PointNotFound',
    # Partner's system errors: code >= 100 && code < 200
    'PartnerError',
    'InvalidRide',
    # Booking errors: code >= 200 && code < 300
    'BookingError',
    'Expired',
    'InvalidPlace',
    'NoFreePlace',
    'PlaceAlreadyOccupied',
    'InvalidCredentials',
    'OfflinePaymentDisabled',
    'InvalidPrice',
    'ExtendedTicketTypeNotFound',
    'TicketTypeChangeAtBooking',
    'PassengersDataChangeAtBooking',
    'InvalidTicketBlank',
    'RideHasNoSubstitution',
    # Confirmation errors: code >= 300 && code < 400
    'ConfirmError',
    'InvalidOrder',
    'AlreadyConfirmed',
    # Refund errors: code >= 400 && code < 500
    'RefundError',
    'InvalidTicket',
]


class Error(Exception):
    """Base error."""
    code = 0

    def __init__(self, *args, **kwargs):
        super(Error, self).__init__(*args)
        self.context = kwargs.pop('context', None)

    def __str__(self):
        args = self.args
        if self.context:
            args += (self.context,)
        return '\n'.join(map(str, args))


class InvalidIdentifier(Error, ValueError):
    code = 1


class PointNotFound(Error, KeyError):
    code = 2

    def __init__(self, sid_or_uid):
        Error.__init__(self, sid_or_uid)
        self.sid_or_uid = sid_or_uid

    def __str__(self):
        return "Point '{}' was not found".format(self.sid_or_uid)


class PartnerError(Error):
    code = 100

    def __init__(self, message, response, fault=None):
        Error.__init__(self, message)
        self.message = message
        self.response = response
        self.fault = fault or {}


class InvalidRide(Error):
    code = 110


class BookingError(Error):
    code = 200


class Expired(BookingError):
    code = 201


class InvalidPlace(BookingError):
    code = 202


class NoFreePlace(BookingError):
    code = 203


class PlaceAlreadyOccupied(BookingError):
    code = 204


class InvalidCredentials(BookingError):
    code = 205


class OfflinePaymentDisabled(BookingError):
    code = 206


class InvalidPrice(BookingError):
    code = 207


class ExtendedTicketTypeNotFound(BookingError):
    code = 208


# reserved codes in meta and front
# 209, 210, 211


class TicketTypeChangeAtBooking(BookingError):
    code = 212


class PassengersDataChangeAtBooking(BookingError):
    code = 213


class InvalidTicketBlank(BookingError):
    code = 214


class RideHasNoSubstitution(BookingError):
    code = 215


class ConfirmError(Error):
    code = 300


class InvalidOrder(ConfirmError):
    code = 301


class AlreadyConfirmed(ConfirmError):
    code = 302


class RefundError(Error):
    code = 400


class InvalidTicket(RefundError):
    code = 401
