# -*- coding: utf-8 -*-
import string
from datetime import datetime

from flask_restful import fields, marshal

from yabus.common.exceptions import InvalidPrice
from yabus.common.fields.validation import Validatable, ValidationWarning, validate

__all__ = [
    'Constant',
    'Dict',
    'FormattedString',
    'InvBoolean',
    'ISODate',
    'Price',
    'TryInteger',
    'VAT',
    'ConcatString',
    'SeparatedDocumentNumber',

    'Managed',
    'Required',
    'Optional',
    'pillow'
]


class Constant(fields.Raw):
    def __init__(self, default):
        super(Constant, self).__init__(default=default, attribute='__not_existent__')

    def output(self, key, obj):
        return self.default


class Dict(fields.Nested):
    def __init__(self, **kwargs):
        super(Dict, self).__init__({k: self._make_field(f) for k, f in kwargs.items()})

    def output(self, key, obj):
        return marshal(obj, self.nested)

    @staticmethod
    def _make_field(f):
        return f if isinstance(f, fields.Raw) else fields.Raw(attribute=f)


class FormattedString(fields.FormattedString):
    class Formatter(string.Formatter):
        def get_field(self, field_name, args, kwargs):
            first, rest = field_name._formatter_field_name_split()
            obj = self.get_value(first, args, kwargs)
            for _, i in rest:
                obj = obj[i]
            return obj, first

    def output(self, key, obj):
        data = fields.to_marshallable_type(obj)
        return FormattedString.Formatter().format(self.src_str, **data)


class InvBoolean(fields.Boolean):
    def format(self, value):
        return not super(InvBoolean, self).format(value)


class ISODate(fields.DateTime):
    def format(self, value):
        value = datetime.strptime(value[:10], '%Y-%m-%d')
        return super(ISODate, self).format(value)


class Price(fields.Float, Validatable):
    def validate(self, value, path):
        if value <= 0:
            msg = self.format_msg(path, "must be > 0, got {}".format(value))
            return [InvalidPrice(msg)]
        return []


class TryInteger(fields.Integer):
    def format(self, value):
        try:
            return super(TryInteger, self).format(value)
        except:
            return self.default


class VAT(fields.String):
    conv = {
        # (value, postpayment)
        (18.0, True): 'nds_18',
        (0.18, True): 'nds_18',
        (18.0, False): 'nds_18_118',
        (0.18, False): 'nds_18_118',
        (10.0, True): 'nds_10',
        (0.10, True): 'nds_10',
        (10.0, False): 'nds_10_110',
        (0.10, False): 'nds_10_110',
        (0.0, True): 'nds_0',
        (0.0, False): 'nds_0',
        (None, True): 'nds_none',
        (None, False): 'nds_none',
    }

    def __init__(self, default=None, postpayment=False, attribute=None):
        super(VAT, self).__init__(VAT.conv[(default, postpayment)], attribute)
        self.postpayment = postpayment

    def format(self, value):
        value = float(value) if value is not None else None
        return super(VAT, self).format(VAT.conv[(value, self.postpayment)])


class ConcatString(fields.Raw):
    sep = ', '

    def __init__(self, *keys):
        fields.Raw.__init__(self)
        self._fields = [fields.String(attribute=x) for x in keys]

    def output(self, key, obj):
        return self.sep.join(filter(None, (x.output(key, obj) for x in self._fields)))


class SeparatedDocumentNumber(ConcatString):
    sep = ' '

    def __init__(self, series_attribute, number_attribute):
        super(SeparatedDocumentNumber, self).__init__(series_attribute, number_attribute)


class Managed(fields.Raw, Validatable):
    def __init__(self, field, required, types=None):
        if isinstance(field, type):
            field = field()
        elif isinstance(field, dict):
            field = Dict(**field)
        self.output = field.output
        self.format = field.format
        self.field = field
        self.required = required
        self.types = types

    def validate(self, value, path):
        if value is None:
            return [ValidationWarning(path, "must not be None")] if self.required else []
        elif self.types is not None and not isinstance(value, self.types):
            return [ValidationWarning(path, Validatable.invalid_type(self.types, type(value)))]
        return validate(self.field, value, path)


class Optional(Managed):
    def __init__(self, field, types=None):
        super(Optional, self).__init__(field, False, types)


class Required(Managed):
    def __init__(self, field, types=None):
        super(Required, self).__init__(field, True, types)


def pillow(cls):
    class SoftPillow(cls):

        def format(self, value):
            try:
                return super(SoftPillow, self).format(value)
            except fields.MarshallingException:
                return self.default

    return SoftPillow
