# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from collections import defaultdict
from itertools import chain, product
from logging import getLogger, Logger

import six

from util.lazy_setuper import LazySetuper
from util.point_key import PointKey, PointType
from yabus.common.exceptions import PointNotFound
from yabus.providers import point_relations_provider, point_matching_provider, supplier_provider

logger = getLogger(__name__)


class PointConverter(LazySetuper):
    """Converter is used to convert IDs from one namespace to another.
    You need to provide mapping table. Back-mapping table will be initialized
    on-the-fly.
    """
    def __init__(self, supplier_code, converter_logger=None):
        super(PointConverter, self).__init__()
        self._logger = converter_logger or logger
        self.supplier_code = supplier_code
        self.mapping = {}
        self.backmapping = {}

    def _setup(self, point_matchings=None, suppliers=None):

        suppliers = suppliers or supplier_provider
        supplier = suppliers.get_by_code(self.supplier_code)
        if supplier is None:
            msg = "can not find supplier for {}".format(self.supplier_code)
            raise ValueError(msg)

        point_matchings = point_matchings or point_matching_provider
        mapping = defaultdict(set)
        for point_matching in point_matchings.itervalues():
            if point_matching.supplier_id == supplier.id and point_matching.HasField("point_key"):
                try:
                    mapping[str(PointKey.load_from_proto(point_matching.point_key))].add(point_matching.supplier_point_id)
                except ValueError as e:
                    self._logger.warn("bad pointmatching: {}".format(e))
        self.mapping = dict((k, frozenset(v)) for k, v in mapping.items())
        self.backmapping = self._mk_backmapping(self.mapping)

    @LazySetuper.setup_required
    def map(self, point_key):
        # type: (t.Text) -> t.FrozenSet[t.Text]
        """Map point with given `point_key` to list of supplier's IDs.
            :param point_key: universal ID
            :return: list of supplier's IDs
            :raise :class:`PointNotFound`: when mapping was not found

        .. seealso::
            * :class:`PointNotFound`
        """
        try:
            supplier_ids = self.mapping[point_key]
        except KeyError:
            self._logger.warn(
                'Mapping does not have the key %s',
                point_key
            )
            raise PointNotFound(point_key)
        return frozenset(supplier_ids)

    @LazySetuper.setup_required
    def deprecated_map(self, point_key):
        """
        Функция добавлена для единообразного интерфейса конвертеров.
        После завершения рефакторинга клиентов нужно убрать в :class:`Converter` и :class:`RelationsConverter`
        :param point_key:
        :return: list of supplier's IDs
        :raise :class:`PointNotFound`: when mapping was not found
        """
        return self.map(point_key)

    @LazySetuper.setup_required
    def backmap(self, supplier_id):
        """Map point with given `supplier_id` to single universal ID.
            :param supplier_id: supplier's ID
            :return: universal ID
            :raise :class:`PointNotFound`: when back-mapping was not found

        .. seealso::
            * :class:`PointNotFound`
        """
        if supplier_id not in self.backmapping:
            self._logger.warn(
                'Back mapping does not have the key %s',
                supplier_id
            )
            raise PointNotFound(supplier_id)
        return self.backmapping[supplier_id]

    @LazySetuper.setup_required
    def gen_map_segments(self, segments):
        for segment in segments:
            from_supplier_id, to_supplier_id = segment
            try:
                yield (
                    self.backmap(from_supplier_id),
                    self.backmap(to_supplier_id),
                )
            except PointNotFound:
                pass

    def _mk_backmapping(self, mapping):
        result = {}
        for point_key, supplier_ids in mapping.items():
            for supplier_id in supplier_ids:
                if supplier_id not in result:
                    result[supplier_id] = point_key
                else:
                    if result[supplier_id] != point_key:
                        self._logger.warn(
                            'Back mapping conflict %s map to %s and %s',
                            supplier_id,
                            result[supplier_id],
                            point_key,
                        )
        return result


class PointRelationConverter(PointConverter):
    """Converter decorator. Extends point_keys lists using their relations.

    .. seealso::
        * :class:`PointConverter`
    """
    def __init__(self, supplier_code, converter_logger=None):
        super(PointRelationConverter, self).__init__(supplier_code, converter_logger)
        self.mapping_to_parents = {}
        self.mapping_to_children = {}
        self.relations_mapping = {}
        self.relations_backmapping = {}
        self.backmapping_to_parents = {}
        self.backmapping_to_children = {}

    def _setup(self, suppliers=None, point_matchings=None, relations_provider=None):
        super(PointRelationConverter, self)._setup(suppliers=suppliers, point_matchings=point_matchings)
        self._was_setup = True

        relations_provider = relations_provider or point_relations_provider
        backmapping_to_parents = defaultdict(set)
        backmapping_to_children = defaultdict(set)
        relations_mapping = defaultdict(set)
        for point_key, supplier_ids in six.iteritems(self.mapping):
            relations_mapping[point_key].update(supplier_ids)
            for extra_point_key in relations_provider.get_parents(point_key):
                relations_mapping[extra_point_key].update(supplier_ids)

            try:
                point_key_type = PointKey.load(point_key).type
            except ValueError:
                logger.error('Bad point_key format: %s', point_key)
                continue
            if point_key_type == PointType.SETTLEMENT:
                children_supplier_ids = frozenset(chain(*(self.map(p) for p in relations_provider.get_children(point_key) if p in self.mapping)))
                self.mapping_to_children[point_key] = children_supplier_ids
                for supplier_id in children_supplier_ids:
                    backmapping_to_parents[supplier_id].add(point_key)
            else:
                parents_supplier_ids = frozenset(chain(*(self.map(p) for p in relations_provider.get_parents(point_key) if p in self.mapping)))
                self.mapping_to_parents[point_key] = parents_supplier_ids
                for supplier_id in parents_supplier_ids:
                    backmapping_to_children[supplier_id].add(point_key)

        relations_backmapping = defaultdict(set)
        for point_key, supplier_ids in six.iteritems(relations_mapping):
            for supplier_id in supplier_ids:
                relations_backmapping[supplier_id].add(point_key)

        self.relations_mapping = dict(relations_mapping)
        self.relations_backmapping = dict(relations_backmapping)
        self.backmapping_to_parents = dict((k, frozenset(v)) for k, v in backmapping_to_parents.items())
        self.backmapping_to_children = dict((k, frozenset(v)) for k, v in backmapping_to_children.items())

    @LazySetuper.setup_required
    def map(self, point_key, use_relations=False):
        try:
            return super(PointRelationConverter, self).map(point_key)
        except PointNotFound:
            if use_relations and point_key in self.relations_mapping:
                return frozenset(self.relations_mapping[point_key])
            raise

    @LazySetuper.setup_required
    def deprecated_map(self, point_key):
        """

        :param point_key:
        :return:
        .. deprecated::
            Коннектор должен при работе с маппингами в явном виде указывать,
            какие маппинги (прямые, сужение, расширение) ему нужны.
            Use :func:`map`, :func:`map_to_parents`, :func:`map_to_children` instead.
        """
        try:
            return frozenset(self.relations_mapping[point_key])
        except KeyError:
            raise PointNotFound(point_key)

    @LazySetuper.setup_required
    def map_to_parents(self, point_key):
        return self.mapping_to_parents.get(point_key, frozenset())

    @LazySetuper.setup_required
    def map_to_children(self, point_key):
        return self.mapping_to_children.get(point_key, frozenset())

    @LazySetuper.setup_required
    def backmap_to_parents(self, supplier_point_id):
        return self.backmapping_to_parents.get(supplier_point_id, frozenset())

    @LazySetuper.setup_required
    def backmap_to_children(self, supplier_point_id):
        return self.backmapping_to_children.get(supplier_point_id, frozenset())

    @LazySetuper.setup_required
    def deprecated_relations_backmap(self, supplier_point_id):
        """

        :param supplier_point_id:
        :return:
        .. deprecated::
        """
        return self.relations_backmapping.get(supplier_point_id, ())

    @LazySetuper.setup_required
    def gen_map_segments(self, segments):
        """

        :param segments:
        :return:
        .. deprecated::
            Generate segments with :func:`backmap`, :func:`backmap_to_parents`, :func:`backmap_to_children`
        """
        result = set()
        for supplier_id_from, supplier_id_to in segments:
            point_keys_from = self.deprecated_relations_backmap(supplier_id_from)
            point_keys_to = self.deprecated_relations_backmap(supplier_id_to)
            result.update(product(point_keys_from, point_keys_to))
        return list(result)
