# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from travel.proto.dicts.buses.common_pb2 import EPointKeyType

from yabus.providers import point_matching_provider
from yabus.providers import supplier_provider
from logging import getLogger

logger = getLogger(__name__)


def coherence(supplier_code, from_uid, to_uid, from_sid, to_sid, date, rides):
    """
        :param supplier_code: supplier code in admin db
        :param from_uid: request's departure point's universal ID
        :param to_uid: request's arrival point's universal ID
        :param from_sid: request's departure point's supplier ID
        :param to_sid: request's arrival point's supplier ID
        :param date: request's departure date
        :param rides: list of :class:`yabus.common.entities.Ride`

        .. seealso::
            https://st.yandex-team.ru/BUSES-1602
            * :class:`yabus.common.entities.Ride`
    """
    supplier = supplier_provider.get_by_code(supplier_code)
    from_point_matching = point_matching_provider.get(supplier.id, from_sid)
    if from_point_matching is None:
        logger.error('point_matching_provider has no data for (%s, %s)', supplier.id, from_sid)
    from_is_supplier_station = from_point_matching and from_point_matching.type == EPointKeyType.POINT_KEY_TYPE_STATION
    from_point_key_is_station = from_point_matching and from_point_matching.point_key.type == EPointKeyType.POINT_KEY_TYPE_STATION
    from_is_station = from_uid.startswith('s')

    to_point_matching = point_matching_provider.get(supplier.id, to_sid)
    if to_point_matching is None:
        logger.error('point_matching_provider has no data for (%s, %s)', supplier.id, from_sid)
    to_is_supplier_station = to_point_matching and to_point_matching.type == EPointKeyType.POINT_KEY_TYPE_STATION
    to_point_key_is_station = to_point_matching and to_point_matching.point_key.type == EPointKeyType.POINT_KEY_TYPE_STATION
    to_is_station = to_uid.startswith('s')
    departure_prefix = date.strftime('%Y-%m-%d')

    def check_ride(ride):
        if from_is_station and not from_is_supplier_station and not from_point_key_is_station and from_uid != ride['from']['id']:
            return False
        if from_is_station and from_is_supplier_station and from_point_key_is_station and from_uid != ride['from']['id']:
            return False
        if to_is_station and not to_is_supplier_station and not to_point_key_is_station and to_uid != ride['to']['id']:
            return False
        if to_is_station and to_is_supplier_station and to_point_key_is_station and to_uid != ride['to']['id']:
            return False
        if not ride['departure'].startswith(departure_prefix):
            return False
        return True

    return filter(check_ride, rides)
