# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from os.path import join

import requests
from requests.adapters import DEFAULT_POOLSIZE, HTTPAdapter

import config
from yabus.common.exceptions import PartnerError
from yabus.util import monitoring
from yabus.util.connector_response import log_response
from yabus.util.connector_context import connector_context

requests.packages.urllib3.disable_warnings()


class Session(requests.Session):
    CONNECTION_TIMEOUT = 3.05  # slightly larger than a multiple of 3
    READ_TIMEOUT = config.REQUEST_READ_TIMEOUT
    MAX_RETRIES = 3

    def __init__(self, url, pool_connections=DEFAULT_POOLSIZE, max_retries=None):
        super(Session, self).__init__()
        self.url = url
        self.mount(self.url, HTTPAdapter(
            pool_connections=pool_connections,
            max_retries=max_retries or self.MAX_RETRIES,
        ))

    def request(self, method, path, **kwargs):
        """Makes HTTP request.
            :param method: GET or POST
            :param path: URI path
            :param kwargs: key-value arguments
            :return: response object
            :raise :class:`HttpError`:

        .. seealso::
            * :class:`HttpError`
        """
        url = join(self.url, path)
        if kwargs.get("timeout") is None:
            kwargs["timeout"] = (self.CONNECTION_TIMEOUT, self.READ_TIMEOUT)
        monitoring_path = kwargs.pop("monitoring_path", None) or path
        monitoring_labels = {"client": "http", "method": method, "path": monitoring_path}
        monitoring.count_request(monitoring_labels)
        response = super(Session, self).request(method, url, **kwargs)
        monitoring.count_response(response, monitoring_labels)
        if connector_context and connector_context.explainer:
            connector_context.explainer.collect(response)
        log_response(response)
        try:
            response.raise_for_status()
        except requests.exceptions.HTTPError as e:
            raise PartnerError(e.message, e.response)
        return response
