# -*- coding: utf-8 -*-
from datetime import datetime

from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.ecolines.converter import point_converter
from yabus.ecolines.identifiers import RideID


class Ride(common.Ride):
    @cfields.id_wrapper(RideID)
    class RideID(cfields.Dict):
        pass

    class Status(common.Ride.Status):
        default_status = 1

    class Endpoint(fields.Raw):
        def __init__(self, *args, **kwargs):
            fields.Raw.__init__(self, *args, **kwargs)
            self._point_key_field = common.Ride.Point(converter=point_converter, attribute=self.attribute)
            self._supplier_point_id = fields.String(attribute=self.attribute)

        def output(self, key, obj):
            stations = obj['stations']
            return {
                'id': self._point_key_field.output(key, obj),
                'desc': stations.get(fields.Raw.output(self, key, obj)),
                'supplier_id': self._supplier_point_id.output(key, obj)
            }

    class DateTime(fields.DateTime):
        def format(self, value):
            value = datetime.strptime(value, '%Y-%m-%d %H:%M')
            return super(Ride.DateTime, self).format(value)

    @cfields.converter
    class Price(fields.Float):
        def conv(self, value):
            return float(value) / 100

    fields = {
        'connector': cfields.Constant('ecolines'),
        'partner': cfields.Constant('ecolines'),
        'partnerName': cfields.Constant('Ecolines'),
        '@id': RideID(ride_sid='id'),
        'status': Status(attribute='__not_existent__'),
        'carrier': cfields.Constant('Ecolines'),
        'from': Endpoint(attribute='outbound.origin'),
        'to': Endpoint(attribute='outbound.destination'),
        'departure': DateTime(dt_format='iso8601', attribute='outbound.departure'),
        'arrival': DateTime(dt_format='iso8601', attribute='outbound.arrival'),
        'price': Price(attribute='fare'),
        'freeSeats': fields.Integer(attribute='__not_existent__', default=-1),
        'refundConditions': cfields.Constant(
            'Условия возврата размещены на сайте перевозчика: ' +
            'https://ecolines.net/ru/ru/info/pravila#p6'
        )
    }
