# -*- coding: utf-8 -*-
from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.ecolines.entities.ride import Ride


class RideDetails(common.RideDetails):
    @cfields.converter
    class TicketType(common.RideDetails.TicketType):
        conv = {
            1:  1,
            10: 2,
        }

    @cfields.converter
    class DocumentType(common.RideDetails.DocumentType):
        conv = {
            '00': common.RideDetails.DOC_TYPE_PASSPORT,
            '04': common.RideDetails.DOC_TYPE_BIRTH_CERT,
            '02': common.RideDetails.DOC_TYPE_INTERNATIONAL_PASSPORT,
            '03': common.RideDetails.DOC_TYPE_FOREIGN_PASSPORT,
        }

    @cfields.converter
    class GenderType(common.RideDetails.GenderType):
        conv = {
            'MALE': 1,
            'FEMALE': 2,
        }

    @cfields.converter
    class SeatStatus(common.RideDetails.SeatStatus):
        conv = int

    _place_type = {'id': 1, 'name': 'seat'}
    _doc_types = [{'code': str(c), 'type': t} for c, t in DocumentType.variants()]
    _gender_types = [{'code': c, 'type': t} for c, t in GenderType.variants()]
    _citizenships = [{'code': c, 'name': c} for c in common.citizenships]

    fields = {
        'docTypes': fields.Raw(default=_doc_types, attribute='__not_existent__'),
        'ticketTypes': fields.List(fields.Nested({
            'code': fields.String(attribute='tariff'),
            'type': TicketType(attribute='tariff'),
            'price': Ride.Price(attribute='amount'),
        })),
        'genderTypes': fields.Raw(default=_gender_types, attribute='__not_existent__'),
        'citizenships': fields.Raw(default=_citizenships, attribute='__not_existent__'),
        'seats': fields.List(fields.Nested({
            'code': fields.String(attribute='id'),
            'number': fields.String(attribute='id'),
        })),
        'map': fields.List(fields.Nested({
            'x': fields.Integer(attribute='column'),
            'y': fields.Integer(attribute='row'),
            'type': fields.Raw(default=_place_type, attribute='__not_existent__'),
            'status': SeatStatus(attribute='busy'),
            'number': fields.String(attribute='id'),
        })),
        'from': Ride.Endpoint(attribute='leg.origin'),
        'to': Ride.Endpoint(attribute='leg.destination'),
        'departure': Ride.DateTime(dt_format='iso8601', attribute='leg.departure'),
        'arrival': Ride.DateTime(dt_format='iso8601', attribute='leg.arrival'),
    }

    @classmethod
    def format(cls, value):
        value['seats'] = [x for x in value['seats'] if not x['busy']]
        value['ticketTypes'] = [x for x in value['ticketTypes'] if x.get('discount') is None]
        return super(RideDetails, cls).format(value)
