# -*- coding: utf-8 -*-
from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.etraffic.converter import point_converter
from yabus.etraffic.identifiers import RideID


class Ride(common.Ride):
    @cfields.id_wrapper(RideID)
    class RideID(cfields.Dict):
        pass

    @cfields.converter
    class Status(common.Ride.Status):
        conv = {
            1: 1,
            2: 2,
            11: 6,
        }

    class Name(fields.Raw):
        def format(self, value):
            return value.replace(' - ', u' — '.encode('utf-8'))

    class CarrierID(fields.String):
        def __init__(self, inn_attr, name_attr):
            self.carrier_inn = fields.String(attribute=inn_attr)
            self.carrier_name = fields.String(attribute=name_attr)

        def output(self, key, obj):
            return self.carrier_inn.output(key, obj) or self.carrier_name.output(key, obj)

    class Fee(fields.Float):
        def __init__(self, price, supplier_price):
            self.price = fields.Float(attribute=price)
            self.supplier_price = fields.Float(attribute=supplier_price)

        def output(self, key, obj):
            price = self.price.output(key, obj)
            supplier_price = self.supplier_price.output(key, obj)
            return price - supplier_price

    fields = {
        'connector': cfields.Constant('etraffic'),
        'partner': cfields.Constant('etraffic'),
        'partnerName': cfields.Constant('Е-траффик'),
        'carrierID': CarrierID(inn_attr='carrierInn', name_attr='carrier'),
        '@id': RideID(ride_sid='uid'),
        'status': Status(attribute='status.id'),
        'number': fields.String(attribute='num'),
        'name': Name,
        'bus': fields.String(attribute='busInfo'),
        'from': common.Ride.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='dispatchPointId'),
            desc='dispatchStationName',
            supplier_id=fields.String(attribute='dispatchPointId')),
        'to': common.Ride.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='arrivalPointId'),
            desc='arrivalStationName',
            supplier_id=fields.String(attribute='arrivalPointId')),
        'departure': fields.DateTime(attribute='dispatchDate', dt_format='iso8601'),
        'arrival': fields.DateTime(attribute='arrivalDate', dt_format='iso8601'),
        'fee': Fee(price='price', supplier_price='supplierPrice'),
        'freeSeats': fields.Integer(attribute='freeSeatCount'),
    }

    @classmethod
    def format(cls, value):
        instance = super(Ride, cls).format(value)
        if isinstance(instance, list):
            for ride, ride_value in zip(instance, value):
                cls._add_own_carrier(ride, ride_value)
        elif isinstance(instance, dict):
            cls._add_own_carrier(instance, value)
        return instance

    @classmethod
    def _add_own_carrier(cls, instance, value):
        if instance.get('carrierModel'):
            return instance

        inn = cls._get_inn(value.get('carrierInn'))
        if not inn:
            return instance

        instance['carrierModel'] = {
            'name': value.get('carrier'),
            'inn': inn,
            'timetable': common.Ride.default_carrier_timetable,
        }

    @classmethod
    def _get_inn(cls, value):
        if not value:
            return None
        if value.startswith('ИНН'):
            inn = value.split(' ')
            return inn[1] if len(inn) == 2 else None
        if value.isdigit():
            return value
        return None
