# -*- coding: utf-8 -*-
import re
from itertools import groupby

from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.common.exceptions import ExtendedTicketTypeNotFound, NoFreePlace
from yabus.etraffic.converter import point_converter
from yabus.etraffic.entities.ride import Ride
from yabus.util import deduplicate


class RideDetails(common.RideDetails):
    @cfields.converter
    class GenderType(common.RideDetails.GenderType):
        conv = {
            'M': 1,
            'F': 2,
        }

    @cfields.converter
    class TicketType(common.RideDetails.TicketType):
        conv = {
            'Полный': 1,
            'Полный (Билет АТП)': 1,
            'Полный билет': 1,
            'Пассажирский': 1,
            'Детский': 2,
            'Детский (Билет АТП)': 2,
            'Детский билет с 5 до 12 лет': 2,
            'Детский билет с 0 до 12 лет': 2,
            'Багажный': 3,
        }

    class DocumentType(common.RideDetails.OTBDocType):
        def format(self, value):
            try:
                value = int(value)
            except (TypeError, ValueError):
                return self.default
            return super(RideDetails.DocumentType, self).format(value)

    class SeatNumber(fields.String):
        def format(self, value):
            try:
                value = re.findall(r'(Место )?(\d+)', value)[0][1]
            except IndexError:
                return self.default
            return super(RideDetails.SeatNumber, self).format(value)

    _genderTypes = [{'code': c, 'type': t} for c, t in GenderType.variants()]

    fields = {
        'docTypes': fields.List(fields.Nested({
            'code': fields.String,
            'type': DocumentType(attribute='type'),
        })),
        'genderTypes': fields.Raw(attribute='__not_existent__', default=_genderTypes),
        'citizenships': fields.Raw,
        'ticketTypes': fields.List(fields.Nested({
            'code': fields.String(default=''),
            'type': TicketType(attribute='name'),
            'price': fields.Float,
        })),
        'seats': fields.List(fields.Nested({
            'code': fields.String,
            'number': SeatNumber(attribute='name'),
        })),
        'from': common.RideDetails.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='race.dispatchPointId'),
            desc='race.dispatchStationName'),
        'to': common.RideDetails.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='race.arrivalPointId'),
            desc='race.arrivalStationName'),
        'departure': fields.DateTime(attribute='race.dispatchDate', dt_format='iso8601'),
        'arrival': fields.DateTime(attribute='race.arrivalDate', dt_format='iso8601'),
        'price': fields.Float(attribute='race.price'),
        'fee': Ride.Fee(price='race.price', supplier_price='race.supplierPrice'),
    }

    @staticmethod
    def _deduplicate_ticket_types(ticket_types):
        def type_key(ticket_type):
            return ticket_type['type']['id']

        for _id, type_group in groupby(sorted(ticket_types, key=type_key), key=type_key):
            # selecting ticket type with max price to avoid hypothetical discounted tariffs
            yield max(type_group, key=lambda ticket_type: ticket_type['price'])

    @classmethod
    def format(cls, value):
        if value['race'].get('freeSeatCount') == 0:
            raise NoFreePlace("No free places", context=value)
        instance = super(RideDetails, cls).format(value)
        instance['docTypes'] = list(deduplicate(instance['docTypes'], key=lambda d: d['type']['id']))
        instance['ticketTypes'] = list(cls._deduplicate_ticket_types([
            x for x in instance['ticketTypes'] if (x.get('price') or 0) > 0]))
        instance['seats'] = [seat for seat in instance['seats'] if seat['number']] or None
        return instance


def _make_stop(datetime_attr):
    return fields.Nested({
        'id': cfields.Constant(None),
        'code': fields.Raw(attribute='code'),
        'desc': fields.Raw(attribute='name'),
        'latitude': cfields.Constant(None),
        'longitude': cfields.Constant(None),
        'datetime': fields.DateTime(attribute=datetime_attr, dt_format='iso8601'),
        'priceDiffs': cfields.Constant([])
    })


class AtpRideDetails(RideDetails):
    @cfields.converter
    class TicketType(common.RideDetails.TicketType):
        conv = {
            k + ' (Билет АТП)': v for k, v in RideDetails.TicketType.conv.items()
        }

    fields = dict(RideDetails.fields, **{  # XXX: EntityMeta would fail while making deepcopy of converter
        'ticketTypes': fields.List(fields.Nested({
            'type': TicketType(attribute='name'),
        })),
        'pickupStops': fields.List(_make_stop('dispatchDate')),
        'dischargeStops': fields.List(_make_stop('arrivalDate')),
    })

    @classmethod
    def format(cls, value):
        first_point = {
            'code': value['race']['uid'].split(':')[-2],
            'dispatchDate': value['race']['dispatchDate'],
        }
        value['pickupStops'] = [first_point] + value['stops'][:-1]
        value['dischargeStops'] = value['stops']
        instance = super(AtpRideDetails, cls).format(value)
        if not instance['ticketTypes']:
            raise ExtendedTicketTypeNotFound("ATP ticket was not found", context=value)
        return instance
