# -*- coding: utf-8 -*-
from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.etraffic.entities.ride_details import RideDetails
from yabus.etraffic.identifiers import TicketID


class Ticket(common.Ticket):
    @cfields.id_wrapper(TicketID)
    class TicketID(cfields.Dict):
        pass

    @cfields.converter
    class Status(common.Ticket.Status):
        conv = {
            'B': 1,
            'C': 2,
            'N': 3,
            'E': 4,
            'P': 5,
            'R': 6,
            'S': 7,
        }

    class Barcode(fields.Raw):
        def format(self, value):
            value = super(Ticket.Barcode, self).format(value)
            if value:
                return '{:0>12}'.format(value)
            return value

    @cfields.converter
    class TicketDocumentType(common.RideDetails.DocumentType):
        conv = {
            'Паспорт': common.RideDetails.DOC_TYPE_PASSPORT,
            'Паспорт РФ': common.RideDetails.DOC_TYPE_PASSPORT,
            'Паспорт гражданина РФ': common.RideDetails.DOC_TYPE_PASSPORT,
            'Российский паспорт': common.RideDetails.DOC_TYPE_PASSPORT,
            'Свидетельство о рождении': common.RideDetails.DOC_TYPE_BIRTH_CERT,
            'Свидетельство о рождении РФ': common.RideDetails.DOC_TYPE_BIRTH_CERT,
            'Свидетельство о рожд.(для детей, не достиг.14лет)': common.RideDetails.DOC_TYPE_BIRTH_CERT,
            'Заграничный паспорт': common.RideDetails.DOC_TYPE_INTERNATIONAL_PASSPORT,
            'Заграничный паспорт РФ': common.RideDetails.DOC_TYPE_INTERNATIONAL_PASSPORT,
            'Заграничный паспорт гражданина РФ': common.RideDetails.DOC_TYPE_INTERNATIONAL_PASSPORT,
            'Загранпаспорт гражданина РФ': common.RideDetails.DOC_TYPE_INTERNATIONAL_PASSPORT,
            'Паспорт иностранного гражданина': common.RideDetails.DOC_TYPE_FOREIGN_PASSPORT
        }

    fields = {
        '@id': TicketID(ticket_sid='id', order_sid='__order_sid__'),
        'supplierId': fields.String(attribute='id'),
        'status': Status,
        'url': fields.String,
        'data': {
            'code': fields.String(attribute='ticketCode'),
            'number': fields.String(attribute='ticketNum'),
            'series': fields.String(default=''),
            'barcode': Barcode,
        },
        'priceVat': cfields.VAT(attribute='vat'),
        'passenger': {
            'birthDate': fields.DateTime(attribute='birthday', dt_format='iso8601'),
            'genderType': RideDetails.GenderType(attribute='gender'),
            'docType': TicketDocumentType,
            'docNumber': cfields.SeparatedDocumentNumber(series_attribute='docSeries', number_attribute='docNum'),
            'ticketType': RideDetails.TicketType,
            'seat': fields.String(default='0'),
        },
    }

    @classmethod
    def format(cls, value):
        countries = value.pop('__countries__')
        instance = super(Ticket, cls).format(value)
        for country in countries:
            if country['code'] == instance['passenger']['citizenship']:
                instance['passenger']['citizenship'] = country['name']
        if instance['status']['name'] == 'booked':
            instance['data'] = instance['url'] = None
        return instance
