# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from yabus.common import exceptions
from yabus.util import dump_response


__all__ = [
    'parse_error',
]


def parse_unknown_error(*args, **kwargs):
    raise exceptions.Error(*args, **kwargs)


def parse_expired_error(*args, **kwargs):
    raise exceptions.Expired(*args, **kwargs)


def parse_forbidden_error(*args, **kwargs):
    raise exceptions.Error(*args, **kwargs)


def parse_not_found_error(*args, **kwargs):
    raise exceptions.Error(*args, **kwargs)


def parse_remote_error(faultstring, *args, **kwargs):
    if 'место занято' in faultstring:
        raise exceptions.PlaceAlreadyOccupied(faultstring, *args, **kwargs)
    if 'место не найдено' in faultstring:
        raise exceptions.InvalidPlace(faultstring, *args, **kwargs)
    raise exceptions.Error(faultstring, *args, **kwargs)


def parse_unavailable_error(*args, **kwargs):
    raise exceptions.Error(*args, **kwargs)


def parse_unsupported_error(*args, **kwargs):
    raise exceptions.Error(*args, **kwargs)


def parse_validation_error(faultstring, *args, **kwargs):
    if 'Ошибка в поле' in faultstring:
        raise exceptions.InvalidCredentials(faultstring, *args, **kwargs)
    raise exceptions.Error(faultstring, *args, **kwargs)


type_to_parser = {
    'EXPIRED': parse_expired_error,  # Истек срок действия операции
    'FORBIDDEN': parse_forbidden_error,  # Операция запрещена
    'NOT_FOUND': parse_not_found_error,  # Объект не найден
    'REMOTE': parse_remote_error,  # Ошибка при удаленном вызове
    'UNAVAILABLE': parse_unavailable_error,  # Сервис недоступен
    'UNSUPPORTED': parse_unsupported_error,  # Операция не поддерживается
    'VALIDATION': parse_validation_error,  # Ошибка валидации
}


def parse_error(func):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except exceptions.PartnerError as e:
            error_type = e.fault.get('detail', {}).get('ServiceException', {}).get('type')
            error_parser = type_to_parser.get(error_type, parse_unknown_error)
            faultstring = e.fault.get('faultstring', e.args[0])
            error_parser(faultstring, context=dump_response(e.response))
    return wrapper
