# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import json
import logging
import tempfile
from datetime import datetime, timedelta

import dateutil.parser
import six
import typing
from gevent import sleep
from six.moves import map

try:
    import pathlib
except ImportError:
    import pathlib2 as pathlib

import config
from yabus.util import monitoring
from yabus.util import timer
from yabus.util.auto_updater import AutoUpdater

if typing.TYPE_CHECKING:
    from yabus.etraffic import Client

logger = logging.getLogger(__name__)

_Segment = typing.Tuple[six.text_type, six.text_type]


class SegmentsDump(
    typing.NamedTuple("SegmentsDump", (("created_at", datetime), ("segments", typing.FrozenSet[_Segment]),))
):
    @classmethod
    def load(cls, path):
        # type: (pathlib.Path) -> typing.Optional[SegmentsDump]
        if not path.exists():
            return None

        raw_segments_dump = json.load(path.open())
        created_at = raw_segments_dump["created_at"]
        return SegmentsDump(
            (
                datetime.utcfromtimestamp(created_at)
                if isinstance(created_at, (float, int))
                else dateutil.parser.isoparse(created_at)
            ),
            frozenset(typing.cast(typing.Iterable[_Segment], map(tuple, raw_segments_dump["dump"]))),
        )

    def save(self, path):
        # type: (pathlib.Path) -> None
        with tempfile.NamedTemporaryFile(delete=False, dir=six.text_type(path.parent), mode="wt") as dumpfile:
            # noinspection PyArgumentList
            json.dump(
                {"created_at": self.created_at.isoformat(), "dump": list(self.segments)},
                dumpfile,
                indent=4,
            )
        if path.exists():
            path.unlink()
        pathlib.Path(dumpfile.name).rename(path)

    @property
    def age(self):
        # type: () -> timedelta
        return datetime.utcnow() - self.created_at


class SegmentsProvider(AutoUpdater):
    _client = None  # type: typing.Optional[Client]
    _segments_dump = None  # type: typing.Optional[SegmentsDump]

    def __init__(self, update_period, dump_filename):
        # type: (timedelta, six.text_type) -> None
        super(SegmentsProvider, self).__init__(period=update_period)
        self._dump_path = pathlib.Path(dump_filename)

    def _update(self):
        # type: () -> None
        assert self._client is not None

        with timer.measure_time() as t:
            logger.info("Fetching segments...")
            segments = self._client.raw_segments()['segments']
            self.store_segments(segments)
            logger.info("Segments updated in %s", timedelta(milliseconds=t.delta))

    def run(self, client):
        # type: (Client) -> None
        self._client = client

        segments_dump = self._segments_dump = SegmentsDump.load(self._dump_path)
        if segments_dump is not None:
            ttl = self._period - segments_dump.age
            logger.info("Segments dump expires in %s", ttl)
            if ttl > timedelta():
                sleep(ttl.total_seconds())

        self.setup()
        self.run_update_loop()

    def store_segments(self, segments):
        # type: (typing.List) -> None
        segments_dump = self._segments_dump = SegmentsDump(datetime.utcnow(), frozenset(segments))
        logger.info("Segments updated. Creating dump...")
        segments_dump.save(self._dump_path)
        logger.info("Dump created")

    def get_segments(self):
        # type: () -> typing.Optional[typing.FrozenSet[_Segment]]
        segments_dump = self._segments_dump
        if segments_dump is None:
            return None

        if segments_dump.age > self._period:
            expiration = segments_dump.age - self._period
            monitoring.set_gauge("etraffic.segments_provider.expiration", expiration.total_seconds())
            logger.warning("Segments dump expired for %s", expiration)

        return segments_dump.segments


segments_provider = SegmentsProvider(
    update_period=config.ETRAFFIC_SEGMENTS_UPDATE_PERIOD, dump_filename=config.ETRAFFIC_SEGMENTS_DUMP_PATH
)
