# -*- coding: utf-8 -*-
from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.ok.converter import point_converter
from yabus.ok.entities.ride import Ride
from yabus.util import deduplicate


class RideDetails(common.RideDetails):
    @cfields.converter
    class GenderType(common.RideDetails.GenderType):
        conv = {
            1: 1,
            0: 2,
        }

    @cfields.converter
    class TicketType(common.RideDetails.TicketType):
        conv = {
            1: 1,
            2: 1,
            5: 2,
            8: 1,
            10: 3,
        }

    @cfields.converter
    class DocumentType(common.RideDetails.DocumentType):
        conv = staticmethod(
            lambda x: {
                1: 1,  # Паспорт
                3: 2,  # Свидетельство о рождении
            }.get(int(x))
        )

    _genderTypes = [{'code': str(c), 'type': t} for c, t in GenderType.variants()]

    fields = {
        'docTypes': fields.List(fields.Nested({
            'code': fields.String(attribute='card_identity_id'),
            'type': DocumentType(attribute='card_identity_id'),
        }), attribute='card_identity_list'),
        'genderTypes': fields.Raw(attribute='__not_existent__', default=_genderTypes),
        'citizenships': fields.Raw,
        'refundConditions': fields.String(attribute='ride.ticket_refund_info'),
        'ticketTypes': fields.List(fields.Nested({
            'code': fields.String(attribute='tariff_id'),
            'type': TicketType(attribute='tariff_id'),
            'price': fields.Float(attribute='price_total'),
            'fee': fields.Float(attribute='price_markup'),
        }), attribute='ride.tariff_list'),
        'from': common.RideDetails.Endpoint(
            id=Ride.Point(converter=point_converter, attribute='ride.station_start.station_id'),
            desc=cfields.ConcatString('ride.station_start.city_title', 'ride.station_start.station_title')),
        'to': common.RideDetails.Endpoint(
            id=Ride.Point(converter=point_converter, attribute='ride.station_end.station_id'),
            desc=cfields.ConcatString('ride.station_end.city_title', 'ride.station_end.station_title')),
        'departure': Ride.DateTime(attribute='ride.datetime_start', dt_format='iso8601'),
        'arrival': Ride.DateTime(attribute='ride.datetime_end', dt_format='iso8601'),
        'price': fields.Float(attribute='ride.price_unitiki'),
        'fee': Ride.Fee(price='ride.price_unitiki', tariff='ride.price_source_tariff'),
    }

    @classmethod
    def format(cls, value):
        instance = super(RideDetails, cls).format(value)

        # have to remove duplicate types, otherwise could get white screen on tickets-www
        instance['ticketTypes'] = list(deduplicate(instance['ticketTypes'], key=lambda x: x['type']['id']))

        return instance
