# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import datetime
import logging

import six
import typing
from flask_restful import fields
from six.moves import zip

from yabus import common
from yabus.common import fields as cfields
from yabus.sks.converter import point_converter
from yabus.sks.identifiers import RideID
from yabus.util.humanize import humanize_duration

logger = logging.getLogger(__name__)

_TextDict = typing.Dict[six.text_type, six.text_type]
_RefundConditionTimes = typing.List[typing.Tuple[typing.Optional[int], typing.Optional[int]]]


class Ride(common.Ride):
    @cfields.id_wrapper(RideID)
    class RideID(cfields.Dict):
        pass

    class Status(common.Ride.Status):
        default_status = 1

    class Price(fields.Float):
        def format(self, prices):
            price = next(x['value'] for x in prices if x['type'] == 'adult')
            return super(Ride.Price, self).format(price)

    class FreeSeatsCount(fields.Raw):
        def format(self, value):
            return value.count(',') + (1 if value else 0)

    class RefundConditions(fields.Raw):
        @staticmethod
        def _parse_time(raw_minutes):
            # type: (six.text_type) -> typing.Optional[int]
            if not raw_minutes:
                return None
            return int(raw_minutes)

        @classmethod
        def _check_times(cls, times):
            # type: (_RefundConditionTimes) -> bool

            # valid times are [(None, T1), (T1, 0), (0, T2), (T2, None)]
            # where T1 and T2 are not None and T1 < T2
            # note that each interval should have None or 0 value
            for i, (time_from, time_till) in enumerate(times):
                if time_from is None:
                    if i != 0:
                        return False
                    continue

                if time_till is None:
                    if i != len(times) - 1:
                        return False
                    continue

                if time_from <= time_till:
                    return False

                if time_from != 0 and time_till != 0:
                    return False

                if i > 0 and times[i - 1][1] != time_from:
                    return False

            return True

        @classmethod
        def _parse_times(cls, conditions):
            # type: (typing.Iterable[_TextDict]) -> typing.Optional[_RefundConditionTimes]
            if not conditions:
                return

            times = [
                (cls._parse_time(condition['time_from']), cls._parse_time(condition['time_till']))
                for condition in conditions
            ]
            if not cls._check_times(times):
                return

            return times

        @staticmethod
        def _humanize_minutes(time_till):
            # type: (int) -> six.text_type
            return humanize_duration(datetime.timedelta(minutes=time_till))

        def format(self, conditions):
            # type: (typing.Iterable[typing.Dict[six.text_type, six.text_type]]) -> six.text_type
            times = self._parse_times(conditions)
            if not times:
                logger.warning('Cannot format time intervals of the refund conditions: %s', conditions)
                return self.default

            condition_parts = []
            for condition, (time_from, time_till) in zip(conditions, times):
                return_percent = condition['return_percent']
                if not return_percent:
                    continue

                if time_from is None and time_till is not None and time_till > 0:
                    time_part = 'более {} до отправления'.format(self._humanize_minutes(time_till))
                elif time_from is not None and time_from > 0 and time_till == 0:
                    time_part = 'менее {} до отправления'.format(self._humanize_minutes(time_from))
                elif time_from == 0 and time_till is not None:
                    time_part = 'менее {} после отправления'.format(self._humanize_minutes(time_till))
                elif time_from is not None and time_from < 0 and time_till is None:
                    time_part = 'более {} после отправления'.format(self._humanize_minutes(time_from))
                else:
                    logger.warning("Cannot format condition: %s", condition)
                    return self.default

                condition_parts.append('• {}: {} от тарифа'.format(time_part, return_percent))

            if not condition_parts:
                return self.default

            return '\n'.join(['Сумма возврата зависит от времени:'] + condition_parts)

    fields = {
        'connector': cfields.Constant('sks'),
        'partner': cfields.Constant('sks'),
        'partnerName': cfields.Constant('СКС'),
        'partnerPhone': cfields.Constant('88007700020'),
        'partnerEmail': cfields.Constant('sks-avto@bk.ru'),
        '@id': RideID(
            from_sid='from.id',
            to_sid='to.id',
            date='departure',
            ride_sid='id'),
        'status': Status(attribute='__not_existent__'),
        'number': fields.Raw,
        'name': fields.Raw(attribute='rtname'),
        'carrier': fields.Raw,
        'from': common.Ride.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='from.id'),
            desc='from.desc',
            supplier_id=fields.String(attribute='from.id')),
        'to': common.Ride.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='to.id'),
            desc='to.desc',
            supplier_id=fields.String(attribute='to.id')),
        'price': Price(attribute='price.parts'),
        'freeSeats': FreeSeatsCount(attribute='freeSeatsCount'),
        'refundConditions': RefundConditions(attribute='refund_with_charges', default=common.Ride.refund_conditions),
    }

    @classmethod
    def enrich(cls, ride, stations=None):
        # type: (dict, typing.Optional[dict]) -> dict
        stations = stations or {}
        for key in ['from', 'to']:
            ride[key] = {
                'id': ride[key],
                'desc': stations.get(ride[key]),
            }
        return ride
