# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from itertools import chain, product
from logging import getLogger

from util.point_key import PointKey, PointType
from util.lazy_setuper import LazySetuper

from yabus.common import PointRelationConverter
from yabus.common.exceptions import PointNotFound
from yabus.providers.point_relations_provider import point_relations_provider
from yabus.unitiki.defaults import SUPPLIER_CODE


class UnitikiPointRelationConverter(PointRelationConverter):

    def __init__(self, *args, **kwargs):
        super(UnitikiPointRelationConverter, self).__init__(*args, **kwargs)
        self.relations_provider = None

    def _setup(self, *args, **kwargs):
        super(UnitikiPointRelationConverter, self)._setup(*args, **kwargs)
        self.relations_provider = kwargs.get('relations_provider') or point_relations_provider
        self.mapping = self._transform_mapping(self.mapping)

    def _transform_mapping(self, mapping):
        result = {}
        for rasp_id, supplier_ids in mapping.items():
            supplier_ids = list(supplier_ids)
            # adding cities to stations to be able searching station-city
            for city_point_key in self.relations_provider.get_parents(rasp_id):
                supplier_ids.extend(self.mapping.get(city_point_key, []))
            result[rasp_id] = self._try_to_reduce_supplier_points(rasp_id, supplier_ids)
        return result

    def _try_to_reduce_supplier_points(self, rasp_id, supplier_ids):
        if len(supplier_ids) <= 1 or PointKey.load(rasp_id).type != PointType.SETTLEMENT:
            return supplier_ids

        city_sids = [supplier_id for supplier_id in supplier_ids if supplier_id.startswith('c')]
        if city_sids:
            self._logger.debug('Unitiki: Reduce supplier_ids from %r to %r for %s', supplier_ids, city_sids, rasp_id)
            if len(city_sids) > 1:
                self._logger.warn('Unitiki: Too many city supplier_ids %r for %s', city_sids, rasp_id)
            return city_sids

        return supplier_ids

    @LazySetuper.setup_required
    def gen_map_segments(self, segments):
        for segment in segments:
            from_supplier_id, to_supplier_id = segment
            try:
                rasp_from_id = self.backmap(from_supplier_id)
                rasp_to_id = self.backmap(to_supplier_id)
                for (from_id, to_id) in chain(
                    product(
                        [point_from_id for point_from_id in
                            chain(self.relations_provider.get_children(rasp_from_id), (rasp_from_id,))
                            if point_from_id in self.mapping
                        ],
                        [point_to_id for point_to_id in
                            chain(self.relations_provider.get_children(rasp_to_id), (rasp_to_id,))
                            if point_to_id in self.mapping
                        ]
                    )):
                    yield (from_id, to_id)
            except PointNotFound:
                pass


point_converter = UnitikiPointRelationConverter(SUPPLIER_CODE, converter_logger=getLogger(__name__))
