# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import re
from datetime import datetime

from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.unitiki.converter import point_converter
from yabus.unitiki.identifiers import RideID


class Ride(common.Ride):
    number_name_re = u'№(\\d+)[^_]*_(.*)'

    @cfields.id_wrapper(RideID)
    class RideID(cfields.Dict):
        pass

    class Status(common.Ride.Status):
        default_status = 1  # sale

    class UnderscoreString(fields.Raw):
        def format(self, value):
            return value.replace(u'_', u' ')

    class Number(fields.Raw):
        def format(self, value):
            try:
                value = re.findall(Ride.number_name_re, value)[0][0]
            except Exception:
                return self.default
            return super(Ride.Number, self).format(value)

    class Name(UnderscoreString):
        def format(self, value):
            try:
                value = re.findall(Ride.number_name_re, value)[0][1]
            except Exception:
                pass
            return super(Ride.Name, self).format(value)

    @cfields.converter
    class Point(common.Ride.Point):
        def conv(self, value):
            return 's{}'.format(value)

    class SupplierPoint(fields.String):
        def format(self, value):
            return 's{}'.format(value)

    class EndpointDesc(fields.Raw):
        def __init__(self, *args):
            self.fields = [fields.Raw(attribute=x) for x in args]

        def output(self, key, obj):
            return ', '.join(x for x in (f.output(key, obj) for f in self.fields) if x)

    class DateTime(fields.DateTime):
        def format(self, value):
            value = datetime.strptime(value, '%Y-%m-%d %H:%M:%S')
            return super(Ride.DateTime, self).format(value)

    class Fee(fields.Float):
        CURRENCY_RUB_ID = 1

        def __init__(self, price_attribute, tariff_attribute, tariff_currency_attribute):
            self._price_field = fields.Float(attribute=price_attribute)
            self._tariff_field = fields.Float(attribute=tariff_attribute)
            self._tariff_currency_field = fields.Integer(attribute=tariff_currency_attribute)

        def output(self, key, obj):
            price = self._price_field.output(key, obj) or 0.0
            tariff = self._tariff_field.output(key, obj)
            tariff_currency = self._tariff_currency_field.output(key, obj)
            return (price - tariff) if (tariff and tariff_currency == self.CURRENCY_RUB_ID) else 0.0

    fields = {
        '@id': RideID(ride_sid='ride_segment_id'),
        'status': Status(attribute='__not_existent__'),
        'number': Number(attribute='route_name'),
        'name': Name(attribute='route_name'),
        'carrier': UnderscoreString(attribute='carrier_title'),
        'bus': fields.Raw(attribute='bus.title'),
        'departure': DateTime(attribute='datetime_start', dt_format='iso8601'),
        'arrival': DateTime(attribute='datetime_end', dt_format='iso8601'),
        'price': fields.Float(attribute='price_unitiki'),
        'fee': Fee('price_unitiki', 'price_source_tariff', 'currency_source_id'),
        'freeSeats': fields.Integer(attribute='place_cnt'),
        'ticketLimit': fields.Integer(attribute='buy_place_cnt_max'),
        'canPayOffline': fields.Boolean(attribute='__not_existent__', default=False),
        'bookOnly': fields.Boolean(attribute='__not_existent__', default=False),
        'connector': cfields.Constant('unitiki-new'),
        'partner': cfields.Constant('unitiki-new'),
        'partnerName': cfields.Constant('Unitiki'),
        'from': common.Ride.Endpoint(
            id=Point(converter=point_converter, attribute='station_start.station_id'),
            desc=EndpointDesc('station_start.city_title', 'station_start.station_title'),
            supplier_id=SupplierPoint(attribute='station_start.station_id')),
        'to': common.Ride.Endpoint(
            id=Point(converter=point_converter, attribute='station_end.station_id'),
            desc=EndpointDesc('station_end.city_title', 'station_end.station_title'),
            supplier_id=SupplierPoint(attribute='station_end.station_id')),
    }
