# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.unitiki.converter import point_converter
from yabus.unitiki.entities.ride import Ride


class RideDetails(common.RideDetails):
    @cfields.converter
    class GenderType(common.RideDetails.GenderType):
        conv = {
            1: 1,
            0: 2,
        }

    class DocumentType(common.RideDetails.DocumentType):
        # 1 - Паспорт
        # 3 - Свидетельство о рождении
        # 8 - Загранпаспорт
        # 1 - Иностранный паспорт, идентификатор совпадает с паспортом, код страны иной
        DOC_TYPE_CODES = {
            '1': [common.RideDetails.DOC_TYPE_PASSPORT, common.RideDetails.DOC_TYPE_FOREIGN_PASSPORT],
            '3': [common.RideDetails.DOC_TYPE_BIRTH_CERT],
            '8': [common.RideDetails.DOC_TYPE_INTERNATIONAL_PASSPORT],
        }

        def __init__(self, document_type_attr, citizenship_attr, countries_attr, *args, **kwargs):
            super(RideDetails.DocumentType, self).__init__(*args, **kwargs)
            self._document_type = fields.String(attribute=document_type_attr)
            self._citizenship = fields.String(attribute=citizenship_attr)
            self._countries_attr = countries_attr

        def output(self, key, obj):
            doc_type = self._document_type.output(key, obj)
            if doc_type not in self.DOC_TYPE_CODES:
                return self.default
            # if not passport - we have one value to return
            if doc_type != '1':
                return self.format(self.DOC_TYPE_CODES[doc_type][0])
            # if passport - need to check citizenship
            else:
                citizenship = self._citizenship.output(key, obj)
                countries = obj.get(self._countries_attr)
                country_name = None
                for country in countries:
                    if country['code'] == citizenship:
                        country_name = country['name']
                if country_name == 'RU':
                    return self.format(common.RideDetails.DOC_TYPE_PASSPORT)
                else:
                    return self.format(common.RideDetails.DOC_TYPE_FOREIGN_PASSPORT)

    class DocumentTypes(fields.Raw):
        def format(self, doc_type_list):
            doc_types = []
            for doc_type in doc_type_list:
                supplier_doc_id = doc_type['card_identity_id']
                for buses_doc_id in RideDetails.DocumentType.DOC_TYPE_CODES.get(supplier_doc_id, []):
                    doc_types.append({'code': supplier_doc_id,
                                      'type': common.RideDetails.DocumentType().format(buses_doc_id)})
            return doc_types

    DEFAULT_TICKET_TYPE = common.RideDetails.TicketType().format(common.RideDetails.TICKET_TYPE_FULL)

    class TicketTypes(fields.Raw):
        def __init__(self, price_attribute, tariff_attribute, tariff_currency_attribute):
            self._price_field = fields.Float(attribute=price_attribute)
            self._fee_field = Ride.Fee(price_attribute, tariff_attribute, tariff_currency_attribute)

        def output(self, key, obj):
            return [{
                'code': '0',
                'type': RideDetails.DEFAULT_TICKET_TYPE,
                'price': self._price_field.output(key, obj),
                'fee': self._fee_field.output(key, obj)
            }]

    _genderTypes = [{'code': str(c), 'type': t} for c, t in GenderType.variants()]

    fields = {
        'docTypes': DocumentTypes(attribute='card_identity_list'),
        'ticketTypes': TicketTypes('ride.price_unitiki', 'ride.price_source_tariff', 'ride.currency_source_id'),
        'genderTypes': fields.Raw(attribute='__not_existent__', default=_genderTypes),
        'citizenships': fields.Raw,
        'refundConditions': fields.String(attribute='ride.ticket_refund_info'),
        'from': common.RideDetails.Endpoint(
            id=Ride.Point(converter=point_converter, attribute='ride.station_start.station_id'),
            desc=Ride.EndpointDesc('ride.station_start.city_title', 'ride.station_start.station_title')),
        'to': common.RideDetails.Endpoint(
            id=Ride.Point(converter=point_converter, attribute='ride.station_end.station_id'),
            desc=Ride.EndpointDesc('ride.station_end.city_title', 'ride.station_end.station_title')),
        'departure': Ride.DateTime(attribute='ride.datetime_start', dt_format='iso8601'),
        'arrival': Ride.DateTime(attribute='ride.datetime_end', dt_format='iso8601'),
        'price': fields.Float(attribute='ride.price_unitiki'),
        'fee': Ride.Fee('ride.price_unitiki', 'ride.price_source_tariff', 'ride.currency_source_id'),
    }
