# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import datetime

from flask import current_app as app
from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.unitiki.entities.ride_details import RideDetails
from yabus.unitiki.identifiers import TicketID


class Ticket(common.Ticket):
    @cfields.id_wrapper(TicketID)
    class TicketID(cfields.Dict):
        pass

    @cfields.converter
    class TicketStatus(common.Ticket.Status):
        conv = {
            3: 1,
            2: 6,
            5: 6,
            1: 7,
        }

    class Name(fields.Raw):
        def __init__(self, first=False, middle=False, last=False, *args, **kwargs):
            fields.Raw.__init__(self, *args, **kwargs)
            if first:
                self.part = 1
            if middle:
                self.part = 2
            if last:
                self.part = 0

        def format(self, value):
            try:
                return value.split(' ')[self.part]
            except:
                return self.default

    class Date(fields.DateTime):
        def format(self, value):
            value = datetime.strptime(value, '%Y-%m-%d')
            return super(Ticket.Date, self).format(value)

    fields = {
        '@id': TicketID(ticket_sid='ticket_id', order_sid='operation_id'),
        'supplierId': fields.String(attribute='ticket_id'),
        'status': TicketStatus,
        'url': fields.String,  # lazy
        'price': fields.Float(attribute='price_unitiki'),
        'passenger': {
            'firstName': Name(attribute='name', first=True),
            'middleName': Name(attribute='name', middle=True),
            'lastName': Name(attribute='name', last=True),
            'birthDate': Date(attribute='birthday', dt_format='iso8601'),
            'genderType': RideDetails.GenderType(attribute='gender'),
            'citizenship': fields.String(attribute='citizenship_id'),  # lazy
            'docType': RideDetails.DocumentType(document_type_attr='card_identity_id',
                                                citizenship_attr='citizenship_id',
                                                countries_attr='__countries__'),
            'docNumber': fields.String(attribute='series_number'),
            'ticketType': cfields.Constant(RideDetails.DEFAULT_TICKET_TYPE),
            'seat': fields.String(attribute='position', default='0'),
        },
    }

    @classmethod
    def format(cls, value):
        countries = value.get('__countries__')
        instance = super(Ticket, cls).format(value)
        for country in countries:
            if country['code'] == instance['passenger']['citizenship']:
                instance['passenger']['citizenship'] = country['name']
        return instance
