# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import json
from copy import deepcopy
from itertools import islice, izip, izip_longest

import six
from lxml import etree
from requests_toolbelt.utils import dump


def fmap(func, obj):
    if isinstance(obj, (tuple, list)):
        return [fmap(func, x) for x in obj]
    if isinstance(obj, dict):
        return {fmap(func, k): fmap(func, v) for k, v in obj.items()}
    return func(obj)


def encode_utf8(obj):
    def func(obj):
        if isinstance(obj, six.text_type):
            return obj.encode('utf8')
        return obj
    return fmap(func, obj)


def decode_utf8(obj):
    def func(obj):
        if isinstance(obj, six.binary_type):
            return obj.decode('utf8')
        return obj
    return fmap(func, obj)


def pairwise(seq, zipper=izip):
    return zipper(seq, islice(seq, 1, None))


def merge(xs, ys, func=lambda x, y: x):
    if xs is None:
        return deepcopy(ys)
    if isinstance(xs, dict) and isinstance(ys, dict):
        xs.update({k: merge(xs.get(k), v, func) for k, v in ys.items()})
        return xs
    elif isinstance(xs, (list, tuple)) and isinstance(ys, (list, tuple)):
        return [merge(x, y, func) for x, y in izip_longest(xs, ys)]
    return func(xs, ys)


def dump_response(response):
    def pretty_content(content, content_type):
        if content_type.find('xml') != -1:
            return etree.tostring(etree.fromstring(content), pretty_print=True, encoding='utf-8')
        elif content_type.find('json') != -1:
            return json.dumps(json.loads(content), indent=4, ensure_ascii=False).encode('utf-8')
        return content

    def pretty_request(request):
        try:
            content_type = request.headers.get('Content-Type')
            request.body = pretty_content(request.body, content_type)
        except Exception:
            pass

    def pretty_response(response):
        try:
            content_type = response.headers.get('Content-Type')
            response._content = pretty_content(response.content, content_type)
        except Exception:
            pass

    pretty_response(response)
    if hasattr(response, 'request'):
        pretty_request(response.request)
    try:
        return dump.dump_response(response).decode('utf-8')
    except Exception as ex:
        return '[cannot dump response for url: {}\nstatus: {}\nheaders: {}\nexception: {}]'.\
            format(response.url, response.status_code, response.headers, ex)


class safedict(dict):
    def get(self, key, default=None):
        return super(safedict, self).get(key, default) or default


def deduplicate(xs, key):
    keys = set()
    for x in xs:
        k = key(x)
        if k in keys:
            continue
        keys.add(k)
        yield x


@six.python_2_unicode_compatible
class LazyStr(object):
    _value = None

    def __init__(self, func):
        self._func = func

    def __str__(self):
        value = self._value
        if value is None:
            value = self._value = self._func()
        return value
