# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import typing as t
from functools import partial, wraps

from werkzeug.local import LocalStack, LocalProxy

from yabus.util.explainer import ProtocolExplainer


class ConnectorContext(object):
    connector = None  # type: t.Text
    method = None  # type: t.Text
    explainer = None  # type: t.Optional[ProtocolExplainer]

    def __init__(self, connector):
        # type: (t.Text) -> None
        self.connector = connector

    def __enter__(self):
        _local_stack.push(self)
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        _local_stack.pop()


_local_stack = LocalStack()

connector_context = LocalProxy(lambda: _local_stack.top)  # type: ConnectorContext


def copy_current_connector_context(f):
    context = _local_stack.top
    if context is None:
        raise RuntimeError('No ConnectorContext')

    @wraps(f.func if isinstance(f, partial) else f)
    def wrapper(*args, **kwargs):
        with context:
            return f(*args, **kwargs)

    return wrapper
