# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import typing as t
import json

from requests import Response
from requests.utils import guess_json_utf


class ExplainedException(Exception):

    def __init__(self, explain):
        self.explain = explain
        super(ExplainedException, self).__init__()


class ProtocolExplainer(object):
    EXPLAIN_MAP = [
        ('request_url', lambda r: r.request.url),
        ('request_method', lambda r: r.request.method),
        ('request_headers', lambda r: dict([(h, v) for h, v in r.request.headers.items()])),
        ('request_body', lambda r: ProtocolExplainer._pretty_content(r.request.body)),
        ('response_code', lambda r: r.status_code),
        ('response_headers', lambda r: dict([(h, v) for h, v in r.headers.items()])),
        ('response_body', lambda r: ProtocolExplainer._pretty_content(r.content)),
    ]

    @staticmethod
    def _pretty_content(content):
        # нет проверки формата из заголовка, так как некоторые партнеры его заполняют неверно
        if content:
            try:
                return json.loads(content.decode(guess_json_utf(content)))
            except ValueError:
                pass
        return content

    def __init__(self):
        self._responses = []

    def explain(self):
        # type: () -> t.List[t.Dict[t.Text, t.Any]]
        return [
            {field: getter(r) for field, getter in self.EXPLAIN_MAP}
            for r in self._responses
        ]

    def explained_exception(self):
        return ExplainedException(self.explain())

    def collect(self, response):
        # type: (Response) -> None
        self._responses.append(response)
