# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import typing as t

import gevent
import requests
import six
from library.python.monlib.metric_registry import HistogramType
from travel.library.python.solomon.server import SolomonMetricsServer

from yabus.util.connector_context import connector_context

logger = logging.getLogger(__name__)

_LabelsType = t.Dict[t.Text, t.Text]

_server = None  # type: SolomonMetricsServer

HISTOGRAM_BUCKETS = [0.3, 1, 3, 10, 30]


def init(**kwargs):
    # type: (t.Dict) -> None
    global _server
    _server = SolomonMetricsServer(**kwargs)
    gevent.spawn(_run_server)


def _run_server():
    # type: () -> None
    try:
        _server.run()
    except Exception:
        logger.exception("Server stopped")


def _make_labels(name, labels):
    # type: (t.Text, t.Optional[_LabelsType]) -> _LabelsType
    return dict(labels or {}, name=name)


def set_gauge(name, value, labels=None):
    # type: (t.Text, float, t.Optional[_LabelsType]) -> None
    assert _server is not None, "run init before setting metrics"
    _server.registry.gauge(_make_labels(name, labels)).set(value)


def inc_counter(name, labels=None):
    # type: (t.Text, t.Optional[_LabelsType]) -> None
    assert _server is not None, "run init before setting metrics"
    _server.registry.counter(_make_labels(name, labels)).inc()


def inc_rate(name, labels=None):
    # type: (t.Text, t.Optional[_LabelsType]) -> None
    assert _server is not None, "run init before setting metrics"
    _server.registry.rate(_make_labels(name, labels)).inc()


def collect_to_histogram(name, value, labels=None):
    # type: (t.Text, float, t.Optional[_LabelsType]) -> None
    assert _server is not None, "run init before setting metrics"
    _server.registry.histogram_rate(
        _make_labels(name, labels), HistogramType.Explicit, buckets=HISTOGRAM_BUCKETS
    ).collect(value)


def get_context_labels():
    # type: () -> _LabelsType
    if not connector_context:
        return {}
    return {"connector": connector_context.connector, "connector_method": connector_context.method}


def count_request(extra_labels):
    # type: (_LabelsType) -> None
    inc_rate("partner.request_count", dict(get_context_labels(), **extra_labels))


def count_response(response, extra_labels):
    # type: (requests.Response, _LabelsType) -> None
    context_labels = get_context_labels()
    labels = dict(context_labels, **extra_labels)
    labels["status_code"] = six.text_type(response.status_code)

    inc_rate("partner.request_status_code_count", labels)
    collect_to_histogram("partner.request_duration_seconds", response.elapsed.total_seconds(), context_labels)
