# -*- coding: utf-8 -*-
from flask_restful import fields

from util import timezone as tz
from yabus import common
from yabus.common import fields as cfields
from yabus.yugavtotrans.identifiers import OrderID
from yabus.yugavtotrans.entities.ride import Ride
from yabus.yugavtotrans.entities.ticket import Ticket


class Order(common.Order):
    class SupplierID(fields.Raw):
        def output(self, _, obj):
            return '-'.join(str(t['ticket_id']) for t in obj['tickets'])

    class OrderID(SupplierID):
        def output(self, key, obj):
            value = super(Order.OrderID, self).output(key, obj)
            return OrderID(tickets_id=value).dumps()

    class ExpirationDateTime(Ride.DateTime):
        def parse(self, value):
            value = super(Order.ExpirationDateTime, self).parse(value)
            return value.replace(tzinfo=tz.gettz('Europe/Moscow'))  # tzinfo=MSK - specified by partner

    class TotalPrice(Ride.Price):
        def output(self, key, obj):
            return self.format(sum(int(t['price']) for t in obj['tickets'] if t['price']))

    fields = {
        'partner': cfields.Constant('yugavtotrans'),
        'partnerName': cfields.Constant('ЮгАвтоТранс'),
        '@id': OrderID,
        'supplierId': SupplierID,
        'expirationDateTime': ExpirationDateTime(attribute='reservations_time', dt_format='iso8601'),
        'price': TotalPrice,
        'status': Ticket.Status,
        'tickets': fields.List(Ticket),
    }

    @classmethod
    def format(cls, value):
        value.update({k: value['tickets'][0].get(k) for k in ['reservations_time', 'status']})
        return super(Order, cls).format(value)
