# -*- coding: utf-8 -*-
from datetime import datetime

from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.util.carriers import RegisterType, register_type_format
from yabus.yugavtotrans.converter import point_converter
from yabus.yugavtotrans.identifiers import RideID


class Ride(common.Ride):
    @cfields.id_wrapper(RideID)
    class RideID(cfields.Dict):
        pass

    class Status(common.Ride.Status):
        default_status = 1

    class DateTime(fields.DateTime):
        @staticmethod
        def parse(value):
            return datetime.strptime(value, '%d.%m.%Y %H:%M:%S')

        def format(self, value):
            value = self.parse(value)
            if self.dt_format == 'raw':
                return value
            return super(Ride.DateTime, self).format(value)

    @cfields.converter
    class Price(fields.Float):
        def conv(self, value):
            return float(value) / 100

    class FreeSeatsCount(fields.Integer):
        def format(self, value):
            return len(value)

    fields = {
        'connector': cfields.Constant('yugavtotrans'),
        'partner': cfields.Constant('yugavtotrans'),
        'partnerName': cfields.Constant('ЮгАвтоТранс'),
        'partnerEmail': cfields.Constant('yugavtotrans@mail.ru'),
        '@id': RideID(
            trip_sid='trip_id',
            variant_sid='variant_id',
            from_sid='departure_station_id',
            to_sid='arrival_station_id',
            date=DateTime(attribute='time_of_departure', dt_format='raw')),
        'status': Status,
        'name': fields.String(attribute='route_name'),
        'carrier': fields.String(attribute='carrier_name'),
        'carrierID': fields.String(attribute='carrier_id'),
        'bus': fields.String(attribute='bus_model'),
        'from': common.Ride.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='departure_station_id'),
            desc=cfields.ConcatString('departure_station', 'departure_landing'),
            supplier_id=fields.String(attribute='departure_station_id')),
        'to': common.Ride.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='arrival_station_id'),
            desc=cfields.ConcatString('arrival_station', 'arrival_landing'),
            supplier_id=fields.String(attribute='arrival_station_id')),
        'departure': DateTime(attribute='time_of_departure', dt_format='iso8601'),
        'arrival': DateTime(attribute='time_of_arrival', dt_format='iso8601'),
        'price': Price,
        'freeSeats': FreeSeatsCount(attribute='free_place'),
    }

    @classmethod
    def format(cls, value):
        instance = super(Ride, cls).format(value)
        if isinstance(instance, list):
            for ride, ride_value in zip(instance, value):
                cls._add_own_carrier(ride, ride_value)
        elif isinstance(instance, dict):
            cls._add_own_carrier(instance, value)
        return instance

    @classmethod
    def _add_own_carrier(cls, instance, value):
        if instance.get('carrierModel'):
            return

        carrier_provider = value.get('carrier_provider')
        if not carrier_provider:
            return

        carrier_id = value.get('carrier_id')
        carrier = carrier_provider.get_by_id(carrier_id)
        if not carrier:
            return

        inn = carrier.get('inn')
        ogrn = carrier.get('ogrn')
        ogrnip = carrier.get('ogrnip')
        register_type_value = None
        if ogrn:
            register_type_value = register_type_format(RegisterType.OGRN)
        elif ogrnip:
            register_type_value = register_type_format(RegisterType.OGRNIP)

        instance['carrierModel'] = {
            'legalAddress': carrier.get('legal_address'),
            'registerNumber': ogrn or ogrnip,
            'name': carrier.get('name'),
            'actualAddress': carrier.get('address'),
            'legalName': carrier.get('name'),
            'inn': inn,
            'timetable': carrier.get('working_hours') or common.Ride.default_carrier_timetable,
            'registerType': register_type_value
        }
