# -*- coding: utf-8 -*-
from flask_restful import fields

from yabus import common
from yabus.common import fields as cfields
from yabus.yugavtotrans.citizenships import citizenships
from yabus.yugavtotrans.converter import point_converter
from yabus.yugavtotrans.entities.ride import Ride


class RideDetails(common.RideDetails):
    @cfields.converter
    class DocumentType(common.RideDetails.DocumentType):
        conv = {
            '4': 1,   # паспорт РФ
            '10': 2,  # свидетельство о рождении, старое значение - 9
            '7': 3,   # загран паспорт
            '8': 4,   # иностранный паспорт
        }             # не используем: 5 = военный билет, 6 = водительские права

    class TicketTypes(common.RideDetails.TicketType):
        def output(self, _, obj):
            return [
                {
                    'code': str(code),
                    'type': super(RideDetails.TicketTypes, self).format(code),
                    'price': float(obj[key]) / 100
                }
                for code, key in enumerate(['price', 'price_children'], 1)
            ]

    @cfields.converter
    class GenderType(common.RideDetails.GenderType):
        conv = {
            'M': 1,
            'F': 2,
        }

    @cfields.converter
    class Country(fields.Raw):
        conv = citizenships

    class FreeSeats(fields.Raw):
        def format(self, value):
            return [{'code': str(n), 'number': str(n)} for n in value]

    _doc_types = [{'code': c, 'type': t} for c, t in DocumentType.variants()]
    _gender_types = [{'code': c, 'type': t} for c, t in GenderType.variants()]
    _citizenships = [{'code': c, 'name': n} for c, n in Country.variants()]

    fields = {
        'docTypes': fields.Raw(attribute='__not_existent__', default=_doc_types),
        'ticketTypes': TicketTypes,
        'genderTypes': fields.Raw(attribute='__not_existent__', default=_gender_types),
        'citizenships': fields.Raw(attribute='__not_existent__', default=_citizenships),
        'seats': FreeSeats(attribute='free_place'),
        'map': fields.Raw(attribute='__not_existent__'),
        'from': common.RideDetails.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='departure_station_id'),
            desc=cfields.ConcatString('departure_station', 'departure_landing')),
        'to': common.RideDetails.Endpoint(
            id=common.Ride.Point(converter=point_converter, attribute='arrival_station_id'),
            desc=cfields.ConcatString('arrival_station', 'arrival_landing')),
        'departure': Ride.DateTime(attribute='time_of_departure', dt_format='iso8601'),
        'arrival': Ride.DateTime(attribute='time_of_arrival', dt_format='iso8601'),
        'price': Ride.Price,
    }
