package worker

import (
	"context"
	"time"

	"go.uber.org/atomic"
)

type Task interface {
	MaxRPS() float64
	MaxConcurrency() uint32
	Do()
}

type Worker struct {
	task        Task
	concurrency *atomic.Uint32
}

func NewWorker(task Task) *Worker {
	return &Worker{
		task:        task,
		concurrency: atomic.NewUint32(0),
	}
}

func (w *Worker) taskDo() {
	w.concurrency.Inc()
	w.task.Do()
	w.concurrency.Dec()
}

func (w *Worker) Run(ctx context.Context) {
	go func() {
		ticker := time.NewTicker(time.Duration(float64(time.Second) / w.task.MaxRPS()))
		maxConcurrency := w.task.MaxConcurrency()
		for {
			select {
			case <-ticker.C:
				if w.concurrency.Load() < maxConcurrency {
					go w.taskDo()
				}
			case <-ctx.Done():
				return
			}
		}
	}()
}
