# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from argparse import ArgumentParser
from collections import namedtuple
import json
import logging
import subprocess
import time

from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso


LOG = logging.getLogger(__name__)


Task = namedtuple('Task', ('date', 'process', 'stdout', 'stderr'))


class Runner(object):

    def __init__(self, command, options):
        self.command = command

        self.pool_size = options.pool_size
        self.check_delay = options.check_delay

        self.dates_to_process = iter_day(options.date_from, options.date_to)
        self.processed_dates = self.get_processed_dates()
        self.task_pool = set()
        self.source_id = 0

    @staticmethod
    def get_processed_dates():
        try:
            with open('processed_dates.json') as f:
                raw_dates = json.load(f)
        except IOError:
            raw_dates = []
        return set(parse_datetime_iso(d) for d in raw_dates)

    def save_processed_dates(self):
        dates = [str(d) for d in self.processed_dates]
        with open('processed_dates.json', 'w') as f:
            json.dump(dates, f)

    def new_task(self, date_day):
        logging.info('Starting %r task', date_day)

        stdout = open('{}.stdout'.format(date_day), 'w')
        stderr = open('{}.stderr'.format(date_day), 'w')

        command = self.command + [
            '--date-from', str(date_day),
            '--date-to', str(date_day),
            '--lb-source-id', str(self.source_id),
            '--low-priority-order-processing',
        ]

        self.source_id += 1

        process = subprocess.Popen(command, stdout=stdout, stderr=stderr)

        return Task(date_day, process, stdout, stderr)

    @staticmethod
    def close_task(task):
        task.stdout.close()
        task.stderr.close()

    def fill_pool(self):
        while len(self.task_pool) < self.pool_size:
            date = next(self.dates_to_process, None)
            if date is None:
                break
            if date in self.processed_dates:
                logging.info('Task %r already completed', date)
            else:
                self.task_pool.add(self.new_task(date))

    def check_pool(self):
        finished_tasks = list()

        for task in self.task_pool:
            task_result = task.process.poll()
            if task_result is not None:
                logging.info('Task %r finished with exit code %r', task.date, task_result)
                if task_result == 0:
                    self.processed_dates.add(task.date)
                finished_tasks.append(task)

        for task in finished_tasks:
            self.close_task(task)
            self.task_pool.remove(task)

    def run(self):
        logging.info('Processed dates: %r', self.processed_dates)

        try:
            self.fill_pool()
            while self.task_pool:
                time.sleep(self.check_delay)
                self.check_pool()
                self.fill_pool()
        finally:
            self.save_processed_dates()

        logging.info('Processed dates: %r', self.processed_dates)


def main():
    logging.basicConfig(level=logging.INFO, format='%(asctime)s %(message)s')

    parser = ArgumentParser()
    parser.add_argument('--pool-size', type=int, default=100)
    parser.add_argument('--check-delay', type=int, default=10)

    parser.add_argument('--date-from', type=parse_datetime_iso, required=True)
    parser.add_argument('--date-to', type=parse_datetime_iso, required=True)

    options, rest_args = parser.parse_known_args()

    Runner(rest_args, options).run()


if __name__ == '__main__':
    main()
