# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, datetime

from dateutil.relativedelta import relativedelta
from ticket_parser2.api.v1 import TvmClient, TvmClientStatus, TvmApiClientSettings
from yt.wrapper import YtClient

from travel.avia.library.python.marker_helpers import AviaFlightRouteHelper
from travel.avia.library.python.references.partner import PartnerCache
from travel.avia.library.python.references.station import create_station_cache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import AeroflotAviaOrderSnapshot, OrderCurrencyCode, OrderStatus

LOG = get_logger(__name__)


class AeroflotBoyCollector(HttpCollector):
    PARTNER_NAME = 'aeroflot_boy'
    PARTNER_CODE = 'aeroflot'
    ORDER_TYPE = 'AVIA_AEROFLOT'
    BASE_URL = 'http://api-prod.travel-hotels.yandex.net/api/cpa_export/v2/get_cpa_order_snapshots'
    ALLOW_PROXY = False

    REQUEST_TIMEOUT = 20

    STATUS_MAPPING = {
        'CONFIRMED': OrderStatus.CONFIRMED,
        'PENDING': OrderStatus.PENDING,
        'CANCELLED': OrderStatus.CANCELLED,
        'DELETED': OrderStatus.DELETED
    }

    def __init__(self, options):
        super(AeroflotBoyCollector, self).__init__(options)
        use_tvm = options.use_tvm

        tvm_ticket = None
        if use_tvm:
            tvm_secret = options.tvm_secret
            tvm_client_id = options.tvm_client_id
            tvm_service_id = options.tvm_service_id
            tvm_client = self.create_tvm_client(tvm_service_id, tvm_client_id, tvm_secret)
            tvm_ticket = tvm_client.get_service_ticket_for('cpa_export_api')
        LOG.debug('Got tvm ticket: {}'.format(tvm_ticket))
        self.tvm_ticket = tvm_ticket

        self.date_from = datetime.combine(parse_datetime_iso(options.date_from).date(), datetime.min.time())
        self.date_to = datetime.combine(parse_datetime_iso(options.date_to).date(), datetime.max.time())
        self.limit = options.limit

        self.get_raw_snapshots = with_retries(
            self.get_raw_snapshots_once,
            counter=self.metrics,
            key='collector.events.invalid_response',
        )

        yt_client = YtClient(options.yt_proxy, options.yt_token)
        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_CODE)
        self.route_helper = AviaFlightRouteHelper(create_station_cache(yt_client))

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('--use-tvm', action='store_true')
        parser.add_argument('--tvm-secret')
        parser.add_argument('--tvm-client-id', type=int)
        parser.add_argument('--tvm-service-id', type=int)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

        parser.add_argument('--date-from', default=(date.today() - relativedelta(days=3)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())
        parser.add_argument('--limit', default=300)

    @staticmethod
    def create_tvm_client(tvm_service_id, tvm_client_id, tvm_secret):
        destinations = {'cpa_export_api': tvm_service_id}
        settings = TvmApiClientSettings(
            self_client_id=tvm_client_id,
            enable_service_ticket_checking=True,
            enable_user_ticket_checking=False,
            self_secret=tvm_secret,
            dsts=destinations,
        )
        client = TvmClient(settings)
        if client.status != TvmClientStatus.Ok:
            LOG.exception('Bad tvm client status: %s', client.status)
            raise ProcessError(ErrorType.ET_BAD_TVM_CLIENT_STATUS)
        return client

    def _get_snapshots(self):
        LOG.info('Getting order snapshots starting from %r to %r', self.date_from, self.date_to)
        for raw_snapshot_batch in self.get_raw_snapshots_batch():
            for raw_snapshot in raw_snapshot_batch:
                yield self.get_order_snapshot(raw_snapshot)

    def get_raw_snapshots_batch(self):
        has_more = True
        while has_more:
            rsp = self.get_raw_snapshots()
            response = rsp.json()['order_snapshots']
            has_more = rsp.json()['has_more']
            if has_more:
                LOG.info('Will get one more page')
                self.date_from = parse_datetime_iso(response[-1]['updated_at'])
            yield response

    def get_raw_snapshots_once(self):
        params = {
            'order_type': self.ORDER_TYPE,
            'updated_at_from_utc': self.date_from.isoformat(),
            'updated_at_to_utc': self.date_to.isoformat(),
            'limit': self.limit,
        }
        headers = {}
        if self.tvm_ticket is not None:
            headers['X-Ya-Service-Ticket'] = self.tvm_ticket

        rsp = self.request_get(self.base_url, params=params, headers=headers)
        return rsp

    @staticmethod
    def get_mapped_flight(flight):
        return {
            'from': flight['from'],
            'to': flight['to'],
            'departure_dt': parse_datetime_iso(flight['departure_at']),
            'arrival_dt': parse_datetime_iso(flight['arrival_at']),
        }

    def get_trip_info(self, flights_snapshot):
        flights = [self.get_mapped_flight(fl) for fl in flights_snapshot]
        flights = sorted(flights, key=lambda flight: flight['departure_dt'])
        LOG.debug('orig = %s, dst = %s', flights[0]['from'], flights[-1]['to'])
        return self.route_helper.fillin_trip_info({'flights': flights})

    def get_order_snapshot(self, raw_snapshot):
        snapshot = AeroflotAviaOrderSnapshot()

        snapshot.update_partner_order_id(raw_snapshot['partner_order_id'])
        snapshot.source = 'boy'
        snapshot.status = self.STATUS_MAPPING[raw_snapshot['order_status']]
        snapshot.label = raw_snapshot['label']
        snapshot.currency_code = OrderCurrencyCode(raw_snapshot['amount']['currency'])
        snapshot.order_amount = float(raw_snapshot['amount']['value'])
        snapshot.profit_amount = float(raw_snapshot['profit']['value'])
        snapshot.created_at = timestamp(parse_datetime_iso(raw_snapshot['created_at']))
        snapshot.updated_at = timestamp(self.now)

        # avia specific
        pnr = raw_snapshot.get('pnr')
        if pnr is not None:
            snapshot.pnr = pnr
        trip_info = self.get_trip_info(raw_snapshot['flights'])
        snapshot.origin = trip_info['origin']
        snapshot.destination = trip_info['destination']
        snapshot.trip_type = trip_info['trip_type']
        snapshot.date_forward = trip_info['date_forward']
        snapshot.date_backward = trip_info['date_backward']

        snapshot.partner_id = self.partner_id
        snapshot.billing_order_id = self.billing_order_id

        LOG.debug(snapshot.as_dict())

        return snapshot
