# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import date, datetime
import json

from dateutil.relativedelta import relativedelta
from six import text_type
from yt.wrapper import YtClient

from travel.avia.library.python.references.partner import PartnerCache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, AnywayanydayAviaOrderSnapshot

LOG = get_logger(__name__)


class AnywayanydayCollector(HttpCollector):
    PARTNER_NAME = 'anywayanyday'
    PARTNER_CODE = 'awad'
    SOURCE = 'awad'
    BASE_URL = 'https://api.anywayanyday.com/YandexReport'
    REQUEST_TIMEOUT = 60
    STATUS_MAPPING = {
        'booking': OrderStatus.PENDING,
        'paid': OrderStatus.CONFIRMED,
        'cancel': OrderStatus.CANCELLED,
    }
    TRIP_TYPE_MAPPING = {
        'OW': u'oneway',
        'RT': u'roundtrip',
    }

    def __init__(self, options):
        super(AnywayanydayCollector, self).__init__(options)

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self._login = options.login
        self._password = options.password

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_CODE)

        self.get_day_report = with_retries(
            func=self._get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('-u', '--login', required=True)
        parser.add_argument('-p', '--password', required=True)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=-4)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

    def _get_snapshots(self):
        for day_date in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day_date)
            for snapshot in self.get_day_snapshots(day_date):
                if snapshot is None:
                    continue
                yield snapshot

    def get_day_snapshots(self, day_date):
        """
        Собираем данные от партнера и раскладываем в snapshot
        """
        response = self.get_day_report(day_date)
        for snapshot in self.parse_report(response.content):
            yield snapshot

    def _get_day_report_once(self, day_date):
        params = {
            'date_from': day_date.strftime('%Y-%m-%d'),
            'date_to': day_date.strftime('%Y-%m-%d'),
            'partner': self._login,
            'password': self._password,
        }
        LOG.info(
            'Getting data from %s date: %s',
            self.base_url,
            day_date,
        )
        return self.request_get(
            self.base_url,
            params=params,
            timeout=self.REQUEST_TIMEOUT,
        )

    def parse_report(self, content):
        try:
            orders = json.loads(content)
        except:
            LOG.exception('Bad Anywayanyday JSON')
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        for order_dict in orders:
            try:
                order = {
                    'created_at': timestamp(self._convert_date(order_dict.get('created_at'))),
                    'order_amount': float(order_dict.get('price')),
                    'currency_code': OrderCurrencyCode(order_dict.get('currency')),
                    'status': self._parse_status(order_dict.get('status')),
                    'label': order_dict.get('marker'),
                    'origin': order_dict.get('from'),
                    'destination': order_dict.get('to'),
                    'trip_type': self._parse_trip_type(order_dict.get('type')),
                    'partner_id': self.partner_id,
                    'source': self.SOURCE,
                    'billing_order_id': self.billing_order_id,
                }
                snapshot = AnywayanydayAviaOrderSnapshot.from_dict(
                    d=order,
                    convert_type=False,
                    ignore_unknown=True,
                )
                snapshot.update_partner_order_id(text_type(order_dict.get('orderid')))
                yield snapshot
            except Exception:
                LOG.exception('Anywayanyday parse error')
                raise

    def _parse_status(self, partner_status):
        return self.STATUS_MAPPING[partner_status]

    @staticmethod
    def _convert_date(date_str):
        if not date_str:
            return date_str
        return datetime.strptime(date_str, '%Y-%m-%d')

    def _parse_trip_type(self, raw_trip_type):
        return self.TRIP_TYPE_MAPPING.get(raw_trip_type, 'unknown')
