# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import hashlib
from collections import OrderedDict
from datetime import date, datetime, timedelta
from dateutil.relativedelta import relativedelta
from six import text_type

from lxml import etree
from yt.wrapper import YtClient

from travel.avia.library.python.references.partner import PartnerCache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import iter_day, parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import OrderCurrencyCode, OrderStatus, AzimuthAviaOrderSnapshot

LOG = get_logger(__name__)


class AzimuthCollector(HttpCollector):
    PARTNER_NAME = 'azimuth'
    PARTNER_CODE = 'azimuth'
    SOURCE = 'azimuth'
    BASE_URL = 'https://booking.azimuth.aero/bitrix/components/travelshop/ibe.soap/travelshop_booking.php'
    REQUEST_TIMEOUT = 120
    FETCH_INTERVAL_IN_MONTHS = -2
    TRIP_TYPE_MAPPING = {
        'OW': u'oneway',
        'RT': u'roundtrip',
    }

    def __init__(self, options):
        super(AzimuthCollector, self).__init__(options)

        self.date_from = parse_datetime_iso(options.date_from).date()
        self.date_to = parse_datetime_iso(options.date_to).date()

        self._session_token = options.session_token
        self._shared_secret = options.shared_secret

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(self.PARTNER_CODE)

        self.get_day_report = with_retries(
            func=self._get_day_report_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('-s', '--shared-secret', required=True)
        parser.add_argument('-t', '--session-token', required=True)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

        parser.add_argument('--date-from', default=(date.today() + relativedelta(months=cls.FETCH_INTERVAL_IN_MONTHS)).isoformat())
        parser.add_argument('--date-to', default=date.today().isoformat())

    def _get_snapshots(self):
        for day_date in iter_day(self.date_from, self.date_to):
            LOG.info('Getting snapshots for %r', day_date)
            for snapshot in self.get_day_snapshots(day_date):
                yield snapshot

    def get_day_snapshots(self, day_date):
        """
        Собираем данные от партнера и раскладываем в snapshot
        """
        response = self.get_day_report(day_date)
        for snapshot in self.parse_report(response.content):
            yield snapshot

    def _get_day_report_once(self, report_date):
        next_day = report_date + timedelta(days=1)
        dt_fmt = '%d.%m.%Y %H:%M:%S'
        params = OrderedDict((
            ('session_token', self._session_token),
            ('timestamp_from', report_date.strftime(dt_fmt)),
            ('timestamp_to', next_day.strftime(dt_fmt)),
        ))

        m = hashlib.md5()
        m.update(''.join(params.values()) + self._shared_secret)
        params['hash'] = m.hexdigest()

        query_xml = """
            <soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/" xmlns:tais="http://www.tais.ru/">
               <soapenv:Header/>
               <soapenv:Body>
                  <tais:GetOrderListInput>
                     <tais:session_token>{session_token}</tais:session_token>
                     <tais:hash>{hash}</tais:hash>
                     <tais:timestamp_from>{timestamp_from}</tais:timestamp_from>
                     <tais:timestamp_to>{timestamp_to}</tais:timestamp_to>
                  </tais:GetOrderListInput>
               </soapenv:Body>
            </soapenv:Envelope>
        """.format(**params)

        LOG.info(
            'Getting data from %s date: %s',
            self.base_url,
            report_date,
        )

        return self.request_post(
            self.base_url,
            headers={
                'SOAPAction': 'urn:#GetOrderList',
                'Content-Type': 'text/xml; charset=utf-8',
            },
            data=query_xml.encode('utf-8'),
            timeout=self.REQUEST_TIMEOUT,
        )

    def parse_report(self, content):
        try:
            tree = etree.fromstring(content)
        except etree.XMLSyntaxError:
            LOG.exception('Bad Azimuth XML')
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        currency = tree.findtext('.//ns1:currency', namespaces=tree.nsmap)
        for order_element in tree.xpath('//ns1:orders/ns1:GetOrderListOrder', namespaces=tree.nsmap):
            order = {
                'created_at': timestamp(self._convert_dt(
                    order_element.findtext('ns1:booking_timestamp', namespaces=tree.nsmap)
                )),
                'order_amount': float(order_element.findtext('ns1:full_price', namespaces=tree.nsmap)),
                'currency_code': OrderCurrencyCode(currency),
                'status': self._parse_status(order_element),
                'label': order_element.findtext('ns1:partner_data', namespaces=tree.nsmap),
                'origin': order_element.findtext('ns1:departure_point', namespaces=tree.nsmap),
                'destination': order_element.findtext('ns1:arrival_point', namespaces=tree.nsmap),
                'trip_type': self._parse_trip_type(order_element),
                'partner_id': self.partner_id,
                'source': self.SOURCE,
                'billing_order_id': self.billing_order_id,
            }

            snapshot = AzimuthAviaOrderSnapshot.from_dict(
                d=order,
                convert_type=True,
                ignore_unknown=True,
            )
            order_id = order_element.findtext('ns1:order_id', namespaces=tree.nsmap)
            snapshot.update_partner_order_id(text_type(order_id))
            yield snapshot

    def _parse_status(self, order_element):
        if order_element.findtext('ns1:canceled', namespaces=order_element.nsmap) == 'Y':
            return OrderStatus.CANCELLED
        elif order_element.findtext('ns1:finalized', namespaces=order_element.nsmap) == 'Y':
            return OrderStatus.CONFIRMED
        else:
            return OrderStatus.PENDING

    def _parse_trip_type(self, order_element):
        return self.TRIP_TYPE_MAPPING.get(
            order_element.findtext('ns1:owrt', namespaces=order_element.nsmap),
            'unknown'
        )

    @staticmethod
    def _convert_dt(date_time):
        """
            param str date_time: 13.07.2019 14:27:20
        """
        return datetime.strptime(date_time, '%d.%m.%Y %H:%M:%S')
