# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import datetime as dt

from travel.cpa.collectors.lib.yandex_avia_collector import YandexAviaCollector
from travel.cpa.lib.lib_datetime import parse_datetime_iso, timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import BiletdvAviaOrderSnapshot, OrderStatus

LOG = get_logger(__name__)


class BiletdvCollector(YandexAviaCollector):
    PARTNER_NAME = 'biletdv'
    PARTNER_CODE = 'biletdv'

    BASE_URL = 'https://avia.biletdv.ru/DesktopModules/SASPartnerSales/GetPartnerSalesData.aspx'

    SNAPSHOT_CLS = BiletdvAviaOrderSnapshot
    BOOKINGS_PATH = '//Record'
    FILL_AIRPORTS = True

    STATUS_MAPPING = {
        'SALE': OrderStatus.CONFIRMED,
        'REFUND': OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        super(BiletdvCollector, self).__init__(options)

        self.params = {
            'userName': options.login,
            'password': options.password,
        }

    @classmethod
    def configure(cls, parser):
        super(BiletdvCollector, cls).configure(parser)

        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('--login', required=True)
        parser.add_argument('--password', required=True)

    def get_mapped_flight(self, snapshot_dict):
        departure_dt = parse_datetime_iso(snapshot_dict['DepDate'])
        fake_arrival_dt = departure_dt + dt.timedelta(seconds=1)
        return {
            'from': snapshot_dict['DepCity'],
            'to': snapshot_dict['DestCity'],
            'departure_dt': departure_dt,
            'arrival_dt': fake_arrival_dt,
        }

    def get_flights_from_raw_snapshot(self, raw_snapshot):
        snapshot_dict = self.get_node_dict(raw_snapshot)
        return [self.get_mapped_flight(snapshot_dict)]

    def get_order_snapshot(self, raw_snapshot):
        snapshot_dict = self.get_node_dict(raw_snapshot)
        LOG.debug(snapshot_dict)

        airports = self.fill_airports(raw_snapshot)

        snapshot = self.SNAPSHOT_CLS.from_dict(airports, ignore_unknown=True)

        created_at_dt = parse_datetime_iso(snapshot_dict['CreatedUTC'])  # type: dt.datetime
        created_at = timestamp(created_at_dt)
        # У BiletDV уникальным идентификатором заказа является PNR и created_at
        order_id = '{}_{}'.format(snapshot_dict['PNR'], created_at_dt.strftime('%Y-%m-%dT%H:%M:%S'))
        snapshot.update_partner_order_id(order_id)

        snapshot.status = self.STATUS_MAPPING[raw_snapshot.attrib.get('Type')]
        snapshot.label = snapshot_dict['PersonID']
        snapshot.created_at = created_at
        snapshot.updated_at = timestamp(self.now)
        snapshot.order_amount = float(snapshot_dict['TotalSumma'])
        snapshot.partner_id = self.partner_id
        snapshot.billing_order_id = self.billing_order_id

        LOG.debug('Got snapshot %s from dict %s', snapshot.as_dict(), snapshot_dict)
        return snapshot
