# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
from datetime import datetime

from six import text_type
from yt.wrapper import YtClient

from travel.avia.library.python.references.partner import PartnerCache
from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.lib.common import with_retries
from travel.cpa.lib.errors import ErrorType, ProcessError
from travel.cpa.lib.lib_datetime import timestamp
from travel.cpa.lib.lib_logging import get_logger
from travel.cpa.lib.order_snapshot import BiletixAviaOrderSnapshot, OrderCurrencyCode, OrderStatus

LOG = get_logger(__name__)

SOURCE = PARTNER = 'biletix'


class BiletixCollector(HttpCollector):
    PARTNER_NAME = 'biletix'
    BASE_URL = 'http://partners.biletix.ru/api/export_orders_yt/ordersYandex.json'
    REQUEST_TIMEOUT = 90
    STATUS_MAPPING = {
        'booked': OrderStatus.PENDING,
        'paid': OrderStatus.CONFIRMED,
        'cancelled': OrderStatus.CANCELLED,
    }

    def __init__(self, options):
        super(BiletixCollector, self).__init__(options)

        yt_client = YtClient(options.yt_proxy, options.yt_token)

        self.partner = PartnerCache(yt_client)
        self.partner_id, self.billing_order_id = self.partner.partner_id_bundle(PARTNER)

        self.get_raw_data = with_retries(
            func=self._get_raw_data_once,
            counter=self.metrics,
            key='collector.events.invalid_response'
        )

    @classmethod
    def configure(cls, parser):
        parser.add_argument('--base-url', default=cls.BASE_URL)

        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token', default=None)

    def _get_snapshots(self):
        """
        Собираем данные от партнера и раскладываем в snapshot
        Biletix выдаёт данные скопом за последние 62 дня
        """
        r = self.get_raw_data()
        for snapshot in self.parse_report(r.content):
            yield snapshot

    def _get_raw_data_once(self):
        r = self.request_get(
            self.base_url,
            timeout=self.REQUEST_TIMEOUT,
        )
        return r

    def parse_report(self, content):
        try:
            order_list = json.loads(content)
        except ValueError:
            LOG.exception('Bad Biletix JSON')
            raise ProcessError(ErrorType.ET_PARTNER_DATA)

        for order_dict in order_list:
            price, currency = self._parse_price(order_dict.get('price'))
            order = {
                'status': self._parse_status(order_dict.get('status')),
                'label': order_dict.get('marker'),
                'created_at': timestamp(self._convert_dt(order_dict.get('date_create'))),
                'order_amount': float(price),
                'currency_code': currency,
                'origin': order_dict.get('departure'),
                'destination': order_dict.get('arrival'),
                'trip_type': self._parse_owrt(order_dict.get('owrt')),
                'source': SOURCE,
                'partner_id': self.partner_id,
                'billing_order_id': self.billing_order_id,
            }
            snapshot = BiletixAviaOrderSnapshot.from_dict(
                d=order,
                ignore_unknown=True,
            )
            snapshot.update_partner_order_id(text_type(order_dict.get('order_id')))
            yield snapshot

    @staticmethod
    def _parse_price(input_str):
        """
        Разбиваем строчку с ценой на цену и валюту
        :param basestring input_str: строка вида 1123.01 RUR
        :return: кортеж вида (float_price, currency)
        """
        price_str, currency_str = input_str.split(' ')
        price = float(price_str)
        if currency_str == 'RUR':
            currency_str = 'RUB'
        currency = OrderCurrencyCode(currency_str)
        return price, currency

    def _parse_status(self, partner_status):
        return self.STATUS_MAPPING[partner_status]

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%Y-%m-%d %H:%M:%S')

    @staticmethod
    def _parse_owrt(owrt):
        owrt_map = {
            'OW': 'oneway',
            'RT': 'roundtrip',
            'OJ': 'openjaw',
        }
        return owrt_map.get(owrt)
